/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.KtNodeTypes
import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.descriptors.ClassKind
import ksp.org.jetbrains.kotlin.descriptors.Modality
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.WhenMissingCase
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isEnumClass
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isExpect
import ksp.org.jetbrains.kotlin.fir.declarations.utils.modality
import ksp.org.jetbrains.kotlin.fir.expressions.ExhaustivenessStatus
import ksp.org.jetbrains.kotlin.fir.expressions.FirWhenExpression
import ksp.org.jetbrains.kotlin.fir.expressions.impl.FirElseIfTrueCondition
import ksp.org.jetbrains.kotlin.fir.expressions.impl.FirEmptyExpressionBlock
import ksp.org.jetbrains.kotlin.fir.languageVersionSettings
import ksp.org.jetbrains.kotlin.fir.resolve.toRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.types.isBooleanOrNullableBoolean

object FirExhaustiveWhenChecker : FirWhenExpressionChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirWhenExpression) {
        reportNotExhaustive(expression, context, reporter)
        reportElseMisplaced(expression, reporter, context)
        reportRedundantElse(expression, context, reporter)
    }

    private fun reportEmptyThenInExpression(whenExpression: FirWhenExpression, context: CheckerContext, reporter: DiagnosticReporter) {
        val source = whenExpression.source ?: return

        if (source.isIfExpression && whenExpression.usedAsExpression) {
            val thenBranch = whenExpression.branches.firstOrNull()
            if (thenBranch == null || thenBranch.result is FirEmptyExpressionBlock) {
                reporter.reportOn(source, FirErrors.INVALID_IF_AS_EXPRESSION, context)
            }
        }
    }

    private fun reportNotExhaustive(whenExpression: FirWhenExpression, context: CheckerContext, reporter: DiagnosticReporter) {
        val exhaustivenessStatus = whenExpression.exhaustivenessStatus ?: return
        if (exhaustivenessStatus !is ExhaustivenessStatus.NotExhaustive) {
            // whenExpression.isExhaustive is checked as otherwise the constraint is checked below
            reportEmptyThenInExpression(whenExpression, context, reporter)
            return
        }

        val source = whenExpression.source ?: return

        val subjectType = exhaustivenessStatus.subjectType
        val subjectClassSymbol = subjectType?.toRegularClassSymbol(context.session)

        if (whenExpression.usedAsExpression) {
            if (source.isIfExpression) {
                reporter.reportOn(source, FirErrors.INVALID_IF_AS_EXPRESSION, context)
                return
            } else if (source.isWhenExpression) {
                reportNoElseInWhen(reporter, source, whenExpression, subjectClassSymbol, context)
            }
        } else {
            if (subjectClassSymbol == null) return
            val kind = when {
                subjectClassSymbol.modality == Modality.SEALED -> AlgebraicTypeKind.Sealed
                subjectClassSymbol.classKind == ClassKind.ENUM_CLASS -> AlgebraicTypeKind.Enum
                subjectType.isBooleanOrNullableBoolean -> AlgebraicTypeKind.Boolean
                else -> return
            }

            reportNoElseInWhen(reporter, source, whenExpression, subjectClassSymbol, context)
        }
    }

    private fun reportNoElseInWhen(
        reporter: DiagnosticReporter,
        source: KtSourceElement,
        whenExpression: FirWhenExpression,
        subjectClassSymbol: FirRegularClassSymbol?,
        context: CheckerContext,
    ) {
        val description = when (subjectClassSymbol?.isExpect) {
            true -> {
                val declarationType = if (subjectClassSymbol.isEnumClass) "enum" else "sealed"
                " ('when' with expect $declarationType subject cannot be exhaustive without else branch)"
            }
            else -> ""
        }
        reporter.reportOn(source, FirErrors.NO_ELSE_IN_WHEN, whenExpression.missingCases, description, context)
    }

    private val FirWhenExpression.missingCases: List<WhenMissingCase>
        get() = (exhaustivenessStatus as ExhaustivenessStatus.NotExhaustive).reasons

    private enum class AlgebraicTypeKind(val displayName: String) {
        Sealed("sealed class/interface"),
        Enum("enum"),
        Boolean("Boolean")
    }

    private fun reportElseMisplaced(
        expression: FirWhenExpression,
        reporter: DiagnosticReporter,
        context: CheckerContext
    ) {
        val branchesCount = expression.branches.size
        for (indexedValue in expression.branches.withIndex()) {
            val branch = indexedValue.value
            if (branch.condition is FirElseIfTrueCondition && indexedValue.index < branchesCount - 1) {
                reporter.reportOn(branch.source, FirErrors.ELSE_MISPLACED_IN_WHEN, context)
            }
        }
    }

    private fun reportRedundantElse(whenExpression: FirWhenExpression, context: CheckerContext, reporter: DiagnosticReporter) {
        if (whenExpression.exhaustivenessStatus == ExhaustivenessStatus.RedundantlyExhaustive) {
            for (branch in whenExpression.branches) {
                if (branch.source == null || branch.condition !is FirElseIfTrueCondition) continue
                reporter.reportOn(branch.source, FirErrors.REDUNDANT_ELSE_IN_WHEN, context)
            }
        }
    }

    private val KtSourceElement.isIfExpression: Boolean
        get() = elementType == KtNodeTypes.IF

    private val KtSourceElement.isWhenExpression: Boolean
        get() = elementType == KtNodeTypes.WHEN
}
