/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.descriptors.ClassKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.classKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.isExtensionMember
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.isLocalMember
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.getContainingClassSymbol
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.expressions.FirCallableReferenceAccess
import ksp.org.jetbrains.kotlin.fir.expressions.FirQualifiedAccessExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirResolvedQualifier
import ksp.org.jetbrains.kotlin.fir.expressions.unwrapSmartcastExpression
import ksp.org.jetbrains.kotlin.fir.references.resolved
import ksp.org.jetbrains.kotlin.fir.symbols.FirBasedSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirCallableSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirConstructorSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.hasContextParameters
import ksp.org.jetbrains.kotlin.fir.types.*

object FirCallableReferenceChecker : FirQualifiedAccessExpressionChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirQualifiedAccessExpression) {
        if (expression !is FirCallableReferenceAccess) return

        if (expression.hasQuestionMarkAtLHS && expression.explicitReceiver?.unwrapSmartcastExpression() !is FirResolvedQualifier) {
            reporter.reportOn(expression.source, FirErrors.SAFE_CALLABLE_REFERENCE_CALL)
        }

        // UNRESOLVED_REFERENCE will be reported separately.
        val reference = expression.calleeReference.resolved ?: return
        val referredSymbol = reference.resolvedSymbol
        val source = reference.source ?: return
        if (source.kind is KtFakeSourceElementKind) return

        checkReferenceIsToAllowedMember(referredSymbol, source, context, reporter)
        checkCapturedTypeInMutableReference(expression, referredSymbol, source, context, reporter)
    }

    // See FE 1.0 [DoubleColonExpressionResolver#checkReferenceIsToAllowedMember]
    private fun checkReferenceIsToAllowedMember(
        referredSymbol: FirBasedSymbol<*>,
        source: KtSourceElement,
        context: CheckerContext,
        reporter: DiagnosticReporter,
    ) {
        if (referredSymbol is FirConstructorSymbol && referredSymbol.getContainingClassSymbol()?.classKind == ClassKind.ANNOTATION_CLASS) {
            reporter.reportOn(source, FirErrors.CALLABLE_REFERENCE_TO_ANNOTATION_CONSTRUCTOR, context)
        }

        if (referredSymbol is FirCallableSymbol) {
            if (referredSymbol.isExtensionMember && !referredSymbol.isLocalMember) {
                reporter.reportOn(source, FirErrors.EXTENSION_IN_CLASS_REFERENCE_NOT_ALLOWED, referredSymbol, context)
            }

            if (referredSymbol.hasContextParameters && context.languageVersionSettings.supportsFeature(LanguageFeature.ContextParameters)) {
                reporter.reportOn(source, FirErrors.CALLABLE_REFERENCE_TO_CONTEXTUAL_DECLARATION, referredSymbol, context)
            }
        }
    }

    private fun checkCapturedTypeInMutableReference(
        callableReferenceAccess: FirCallableReferenceAccess,
        referredSymbol: FirBasedSymbol<*>,
        source: KtSourceElement,
        context: CheckerContext,
        reporter: DiagnosticReporter
    ) {
        if (!callableReferenceAccess.resolvedType.isKMutableProperty(context.session)) return
        if (referredSymbol !is FirCallableSymbol<*>) return

        val returnType = context.returnTypeCalculator.tryCalculateReturnType(referredSymbol)
        if (returnType.coneType.hasCapture()) {
            reporter.reportOn(source, FirErrors.MUTABLE_PROPERTY_WITH_CAPTURED_TYPE, context)
        }
    }
}
