/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration

import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.SourceElementPositioningStrategies
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.requireFeatureSupport
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirProperty
import ksp.org.jetbrains.kotlin.fir.isCatchParameter
import ksp.org.jetbrains.kotlin.name.SpecialNames

object FirUnnamedPropertyChecker : FirPropertyChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirProperty) {
        if (declaration.name != SpecialNames.UNDERSCORE_FOR_UNUSED_VAR) {
            return
        }

        val isDesugaredComponentCall = declaration.initializer?.source?.kind == KtFakeSourceElementKind.DesugaredComponentFunctionCall

        if (declaration.isVar && !isDesugaredComponentCall) {
            reporter.reportOn(declaration.source, FirErrors.UNNAMED_VAR_PROPERTY)
        }

        if (declaration.delegate != null) {
            reporter.reportOn(declaration.delegate?.source, FirErrors.UNNAMED_DELEGATED_PROPERTY)
        }

        if (!isDesugaredComponentCall && declaration.isCatchParameter != true) {
            declaration.requireFeatureSupport(
                LanguageFeature.UnnamedLocalVariables, context, reporter,
                positioningStrategy = SourceElementPositioningStrategies.NAME_IDENTIFIER,
            )
        }

        if (declaration.initializer == null && declaration.delegate == null && declaration.isCatchParameter != true) {
            reporter.reportOn(declaration.source, FirErrors.MUST_BE_INITIALIZED)
        }
    }
}
