/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration

import ksp.org.jetbrains.kotlin.descriptors.ClassKind
import ksp.org.jetbrains.kotlin.descriptors.isEnumEntry
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirAnonymousObject
import ksp.org.jetbrains.kotlin.fir.declarations.FirClass
import ksp.org.jetbrains.kotlin.fir.declarations.FirRegularClass
import ksp.org.jetbrains.kotlin.fir.declarations.primaryConstructorIfAny
import ksp.org.jetbrains.kotlin.fir.expressions.FirExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirResolvedQualifier
import ksp.org.jetbrains.kotlin.fir.expressions.allReceiverExpressions
import ksp.org.jetbrains.kotlin.fir.expressions.toReference
import ksp.org.jetbrains.kotlin.fir.expressions.unwrapSmartcastExpression
import ksp.org.jetbrains.kotlin.fir.references.FirThisReference
import ksp.org.jetbrains.kotlin.fir.resolve.dfa.cfg.CFGNodeWithSubgraphs
import ksp.org.jetbrains.kotlin.fir.resolve.dfa.cfg.ControlFlowGraph
import ksp.org.jetbrains.kotlin.fir.resolve.dfa.cfg.FunctionCallExitNode
import ksp.org.jetbrains.kotlin.fir.resolve.dfa.cfg.QualifiedAccessNode
import ksp.org.jetbrains.kotlin.fir.resolve.dfa.controlFlowGraph
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.types.resolvedType
import ksp.org.jetbrains.kotlin.fir.resolve.toRegularClassSymbol
import ksp.org.jetbrains.kotlin.utils.addToStdlib.lastIsInstanceOrNull

object FirEnumCompanionInEnumConstructorCallChecker : FirClassChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirClass) {
        val enumClass = when (declaration.classKind) {
            ClassKind.ENUM_CLASS -> (declaration as FirRegularClass).symbol
            ClassKind.ENUM_ENTRY -> context.containingDeclarations.lastIsInstanceOrNull()
            else -> null
        } ?: return
        val companionOfEnum = enumClass.resolvedCompanionObjectSymbol ?: return
        val graph = declaration.controlFlowGraphReference?.controlFlowGraph ?: return
        analyzeGraph(graph, companionOfEnum, enumClass, context, reporter)
        if (declaration.classKind.isEnumEntry) {
            val constructor = declaration.primaryConstructorIfAny(context.session)
            val constructorGraph = constructor?.resolvedControlFlowGraphReference?.controlFlowGraph
            if (constructorGraph != null) {
                analyzeGraph(constructorGraph, companionOfEnum, enumClass, context, reporter)
            }
        }
    }

    private fun analyzeGraph(
        graph: ControlFlowGraph,
        companionSymbol: FirRegularClassSymbol,
        enumClass: FirRegularClassSymbol,
        context: CheckerContext,
        reporter: DiagnosticReporter
    ) {
        for (node in graph.nodes) {
            if (node is CFGNodeWithSubgraphs) {
                for (subGraph in node.subGraphs) {
                    when (subGraph.kind) {
                        ControlFlowGraph.Kind.AnonymousFunctionCalledInPlace,
                        ControlFlowGraph.Kind.PropertyInitializer,
                        ControlFlowGraph.Kind.ClassInitializer -> analyzeGraph(subGraph, companionSymbol, enumClass, context, reporter)
                        ControlFlowGraph.Kind.Class -> {
                            if (subGraph.declaration is FirAnonymousObject) {
                                analyzeGraph(subGraph, companionSymbol, enumClass, context, reporter)
                            }
                        }
                        else -> {}
                    }
                }
            }
            val qualifiedAccess = when (node) {
                is QualifiedAccessNode -> node.fir
                is FunctionCallExitNode -> node.fir
                else -> continue
            }
            val matchingReceiver = qualifiedAccess.allReceiverExpressions
                .firstOrNull { it.unwrapSmartcastExpression().getClassSymbol(context.session) == companionSymbol }
            if (matchingReceiver != null) {
                reporter.reportOn(
                    matchingReceiver.source ?: qualifiedAccess.source,
                    FirErrors.UNINITIALIZED_ENUM_COMPANION,
                    enumClass,
                    context
                )
            }
        }
    }

    private fun FirExpression.getClassSymbol(session: FirSession): FirRegularClassSymbol? {
        return when (this) {
            is FirResolvedQualifier -> {
                this.resolvedType.toRegularClassSymbol(session)
            }
            else -> (this.toReference(session) as? FirThisReference)?.boundSymbol
        } as? FirRegularClassSymbol
    }
}
