/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 *
 * ASM: a very small and fast Java bytecode manipulation framework
 * Copyright (c) 2000-2011 INRIA, France Telecom
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

package ksp.org.jetbrains.kotlin.codegen.optimization.temporaryVals

import ksp.org.jetbrains.kotlin.codegen.optimization.common.FastAnalyzer
import ksp.org.jetbrains.org.objectweb.asm.Opcodes
import ksp.org.jetbrains.org.objectweb.asm.tree.AbstractInsnNode
import ksp.org.jetbrains.org.objectweb.asm.tree.IincInsnNode
import ksp.org.jetbrains.org.objectweb.asm.tree.MethodNode
import ksp.org.jetbrains.org.objectweb.asm.tree.VarInsnNode
import ksp.org.jetbrains.org.objectweb.asm.tree.analysis.Frame
import ksp.org.jetbrains.org.objectweb.asm.tree.analysis.Interpreter
import ksp.org.jetbrains.org.objectweb.asm.tree.analysis.Value

class StoreLoadFrame<V : Value>(val maxLocals: Int) : Frame<V>(maxLocals, 0) {
    override fun execute(insn: AbstractInsnNode, interpreter: Interpreter<V>) {
        when (insn.opcode) {
            in Opcodes.ISTORE..Opcodes.ASTORE -> {
                val varInsn = insn as VarInsnNode
                setLocal(varInsn.`var`, interpreter.copyOperation(varInsn, null))
            }
            in Opcodes.ILOAD..Opcodes.ALOAD -> {
                val varInsn = insn as VarInsnNode
                interpreter.copyOperation(varInsn, this.getLocal(varInsn.`var`))
            }
            Opcodes.IINC -> {
                val iincInsn = insn as IincInsnNode
                interpreter.unaryOperation(iincInsn, this.getLocal(iincInsn.`var`))
            }
        }
    }
}

class FastStoreLoadAnalyzer<V : Value>(
    owner: String,
    method: MethodNode,
    interpreter: Interpreter<V>,
    newFrame: (Int, Int) -> StoreLoadFrame<V> = { nLocals, _ -> StoreLoadFrame(nLocals) }
) : FastAnalyzer<V, StoreLoadFrame<V>>(
    owner, method, interpreter,
    pruneExceptionEdges = false,
    useFastComputeExceptionHandlers = false,
    useFastMergeControlFlowEdge = false,
    newFrame
)
