/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.backend.wasm.lower

import ksp.org.jetbrains.kotlin.backend.common.FileLoweringPass
import ksp.org.jetbrains.kotlin.backend.wasm.WasmBackendContext
import ksp.org.jetbrains.kotlin.descriptors.ClassKind
import ksp.org.jetbrains.kotlin.ir.IrStatement
import ksp.org.jetbrains.kotlin.ir.backend.js.lower.CallableReferenceLowering.Companion.FUNCTION_REFERENCE_IMPL
import ksp.org.jetbrains.kotlin.ir.backend.js.lower.CallableReferenceLowering.Companion.LAMBDA_IMPL
import ksp.org.jetbrains.kotlin.ir.declarations.*
import ksp.org.jetbrains.kotlin.ir.expressions.IrConstructorCall
import ksp.org.jetbrains.kotlin.ir.expressions.IrExpression
import ksp.org.jetbrains.kotlin.ir.expressions.impl.IrGetObjectValueImpl
import ksp.org.jetbrains.kotlin.ir.util.constructedClass
import ksp.org.jetbrains.kotlin.ir.util.hasShape
import ksp.org.jetbrains.kotlin.ir.util.primaryConstructor
import ksp.org.jetbrains.kotlin.ir.visitors.IrElementTransformerVoid
import ksp.org.jetbrains.kotlin.ir.visitors.transformChildrenVoid

/**
 * Turns static callable references into singletons.
 */
class WasmStaticCallableReferenceLowering(val context: WasmBackendContext) : FileLoweringPass {
    override fun lower(irFile: IrFile) {
        irFile.transformChildrenVoid(object : IrElementTransformerVoid() {
            override fun visitClass(declaration: IrClass): IrStatement {
                declaration.transformChildrenVoid()
                if (declaration.isSyntheticSingleton) {
                    declaration.kind = ClassKind.OBJECT
                }
                return declaration
            }

            override fun visitConstructorCall(expression: IrConstructorCall): IrExpression {
                val constructedClass = expression.symbol.owner.constructedClass
                if (!constructedClass.isSyntheticSingleton)
                    return super.visitConstructorCall(expression)

                return IrGetObjectValueImpl(expression.startOffset, expression.endOffset, expression.type, constructedClass.symbol)
            }
        })
    }
}

val IrClass.isSyntheticSingleton: Boolean
    get() = (origin == LAMBDA_IMPL || origin == FUNCTION_REFERENCE_IMPL) && primaryConstructor!!.hasShape(regularParameters = 0)
