/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.sessions

import ksp.com.intellij.openapi.project.Project
import ksp.com.intellij.psi.search.GlobalSearchScope
import ksp.org.jetbrains.kotlin.analysis.api.platform.packages.createPackagePartProvider
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaDanglingFileModule
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaModule
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaSourceModule
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.factories.LLLibrarySymbolProviderFactory
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.projectStructure.moduleData
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.LLModuleWithDependenciesSymbolProvider
import ksp.org.jetbrains.kotlin.fir.SessionConfiguration
import ksp.org.jetbrains.kotlin.fir.deserialization.SingleModuleDataProvider
import ksp.org.jetbrains.kotlin.fir.java.deserialization.OptionalAnnotationClassesProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.FirSymbolProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.firProvider
import ksp.org.jetbrains.kotlin.fir.scopes.FirOverrideChecker
import ksp.org.jetbrains.kotlin.fir.scopes.impl.FirStandardOverrideChecker
import ksp.org.jetbrains.kotlin.fir.scopes.kotlinScopeProvider
import ksp.org.jetbrains.kotlin.fir.session.FirNativeSessionFactory.registerNativeComponents
import ksp.org.jetbrains.kotlin.utils.addIfNotNull

@OptIn(SessionConfiguration::class)
internal class LLFirNativeSessionFactory(project: Project) : LLFirAbstractSessionFactory(project) {
    override fun createSourcesSession(module: KaSourceModule): LLFirSourcesSession {
        return doCreateSourcesSession(module) { context ->
            registerNativeComponents()

            register(
                FirSymbolProvider::class,
                LLModuleWithDependenciesSymbolProvider(
                    this,
                    providers = listOfNotNull(
                        context.firProvider.symbolProvider,
                        context.switchableExtensionDeclarationsSymbolProvider,
                        context.syntheticFunctionInterfaceProvider,
                    ),
                    context.dependencyProvider,
                )
            )
        }
    }

    override fun createResolvableLibrarySession(module: KaModule): LLFirLibraryOrLibrarySourceResolvableModuleSession {
        return doCreateResolvableLibrarySession(module) { context ->
            registerNativeComponents()
            // Resolvable library session for decompiled libraries can miss annotation arguments
            // necessary for correct work of the native overload checker
            register(FirOverrideChecker::class, FirStandardOverrideChecker(this))
            register(
                FirSymbolProvider::class,
                LLModuleWithDependenciesSymbolProvider(
                    this,
                    providers = listOf(
                        context.firProvider.symbolProvider,
                    ),
                    context.dependencyProvider,
                )
            )
        }
    }

    override fun createBinaryLibrarySession(module: KaModule): LLFirLibrarySession {
        return doCreateBinaryLibrarySession(module) {
            registerNativeComponents()
        }
    }

    override fun createDanglingFileSession(module: KaDanglingFileModule, contextSession: LLFirSession): LLFirSession {
        return doCreateDanglingFileSession(module, contextSession) { context ->
            registerNativeComponents()

            register(
                FirSymbolProvider::class,
                LLModuleWithDependenciesSymbolProvider(
                    this,
                    providers = listOfNotNull(
                        firProvider.symbolProvider,
                        context.switchableExtensionDeclarationsSymbolProvider,
                        context.syntheticFunctionInterfaceProvider,
                    ),
                    context.dependencyProvider,
                )
            )
        }
    }

    override fun createProjectLibraryProvidersForScope(
        session: LLFirSession,
        scope: GlobalSearchScope,
    ): List<FirSymbolProvider> {
        val moduleData = session.moduleData
        val moduleDataProvider = SingleModuleDataProvider(moduleData)
        val packagePartProvider = project.createPackagePartProvider(scope)
        return buildList {
            addAll(
                LLLibrarySymbolProviderFactory.fromSettings(project).createNativeLibrarySymbolProvider(
                    session,
                    scope,
                )
            )
            addIfNotNull(
                OptionalAnnotationClassesProvider.createIfNeeded(
                    session,
                    moduleDataProvider,
                    session.kotlinScopeProvider,
                    packagePartProvider
                )
            )
        }
    }
}
