/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.sessions

import ksp.com.intellij.openapi.project.Project
import ksp.com.intellij.psi.search.GlobalSearchScope
import ksp.org.jetbrains.kotlin.analysis.api.platform.packages.createPackagePartProvider
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaDanglingFileModule
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaModule
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaSourceModule
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.factories.LLLibrarySymbolProviderFactory
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.projectStructure.moduleData
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.LLFirJavaSymbolProvider
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.LLModuleWithDependenciesSymbolProvider
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.providers.nullableJavaSymbolProvider
import ksp.org.jetbrains.kotlin.fir.SessionConfiguration
import ksp.org.jetbrains.kotlin.fir.backend.jvm.FirJvmTypeMapper
import ksp.org.jetbrains.kotlin.fir.deserialization.SingleModuleDataProvider
import ksp.org.jetbrains.kotlin.fir.java.JavaSymbolProvider
import ksp.org.jetbrains.kotlin.fir.java.deserialization.OptionalAnnotationClassesProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.FirSymbolProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.firProvider
import ksp.org.jetbrains.kotlin.fir.resolve.scopes.wrapScopeWithJvmMapped
import ksp.org.jetbrains.kotlin.fir.scopes.FirKotlinScopeProvider
import ksp.org.jetbrains.kotlin.fir.scopes.kotlinScopeProvider
import ksp.org.jetbrains.kotlin.fir.session.registerJavaComponents
import ksp.org.jetbrains.kotlin.load.java.createJavaClassFinder
import ksp.org.jetbrains.kotlin.resolve.jvm.modules.JavaModuleResolver
import ksp.org.jetbrains.kotlin.utils.addIfNotNull

@OptIn(SessionConfiguration::class)
internal class LLFirJvmSessionFactory(project: Project) : LLFirAbstractSessionFactory(project) {
    override fun createSourcesSession(module: KaSourceModule): LLFirSourcesSession {
        return doCreateSourcesSession(module, FirKotlinScopeProvider(::wrapScopeWithJvmMapped)) { context ->
            registerJavaComponents(JavaModuleResolver.getInstance(project))
            val javaSymbolProvider = LLFirJavaSymbolProvider(this, context.contentScope)
            register(JavaSymbolProvider::class, javaSymbolProvider)

            register(
                FirSymbolProvider::class,
                LLModuleWithDependenciesSymbolProvider(
                    this,
                    providers = listOfNotNull(
                        context.firProvider.symbolProvider,
                        context.switchableExtensionDeclarationsSymbolProvider,
                        javaSymbolProvider,
                        context.syntheticFunctionInterfaceProvider,
                    ),
                    context.dependencyProvider,
                )
            )

            register(FirJvmTypeMapper::class, FirJvmTypeMapper(this))
        }
    }

    override fun createResolvableLibrarySession(module: KaModule): LLFirLibraryOrLibrarySourceResolvableModuleSession {
        return doCreateResolvableLibrarySession(module) { context ->
            registerJavaComponents(JavaModuleResolver.getInstance(project))
            val javaSymbolProvider = LLFirJavaSymbolProvider(this, context.contentScope)
            register(
                FirSymbolProvider::class,
                LLModuleWithDependenciesSymbolProvider(
                    this,
                    providers = listOf(
                        context.firProvider.symbolProvider,
                        javaSymbolProvider,
                    ),
                    context.dependencyProvider,
                )
            )
            register(JavaSymbolProvider::class, javaSymbolProvider)
            register(FirJvmTypeMapper::class, FirJvmTypeMapper(this))
        }
    }

    override fun createBinaryLibrarySession(module: KaModule): LLFirLibrarySession {
        return doCreateBinaryLibrarySession(module) {
            registerJavaComponents(JavaModuleResolver.getInstance(project))
            register(FirJvmTypeMapper::class, FirJvmTypeMapper(this))
        }
    }

    override fun createDanglingFileSession(module: KaDanglingFileModule, contextSession: LLFirSession): LLFirSession {
        return doCreateDanglingFileSession(module, contextSession) { context ->
            registerJavaComponents(JavaModuleResolver.getInstance(project))

            val contextJavaSymbolProvider = contextSession.nullableJavaSymbolProvider
            if (contextJavaSymbolProvider != null) {
                register(JavaSymbolProvider::class, contextJavaSymbolProvider)
            }

            register(
                FirSymbolProvider::class,
                LLModuleWithDependenciesSymbolProvider(
                    this,
                    providers = listOfNotNull(
                        firProvider.symbolProvider,
                        context.switchableExtensionDeclarationsSymbolProvider,
                        context.syntheticFunctionInterfaceProvider,
                        contextJavaSymbolProvider
                    ),
                    context.dependencyProvider
                )
            )

            register(FirJvmTypeMapper::class, FirJvmTypeMapper(this))
        }
    }

    override fun createProjectLibraryProvidersForScope(
        session: LLFirSession,
        scope: GlobalSearchScope,
    ): List<FirSymbolProvider> {
        val moduleData = session.moduleData
        val moduleDataProvider = SingleModuleDataProvider(moduleData)
        val packagePartProvider = project.createPackagePartProvider(scope)
        return buildList {
            val firJavaFacade = LLFirJavaFacadeForBinaries(session, project.createJavaClassFinder(scope))
            val deserializedSymbolProviderFactory = LLLibrarySymbolProviderFactory.fromSettings(project)
            addAll(
                deserializedSymbolProviderFactory.createJvmLibrarySymbolProvider(
                    session,
                    firJavaFacade,
                    packagePartProvider,
                    scope,
                )
            )
            addIfNotNull(
                OptionalAnnotationClassesProvider.createIfNeeded(
                    session,
                    moduleDataProvider,
                    session.kotlinScopeProvider,
                    packagePartProvider
                )
            )
        }
    }
}
