/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */
package ksp.org.jetbrains.kotlin.analysis.decompiler.konan

import ksp.com.intellij.openapi.fileTypes.FileType
import ksp.com.intellij.openapi.fileTypes.FileTypeRegistry
import ksp.com.intellij.openapi.vfs.VirtualFile
import ksp.com.intellij.psi.PsiManager
import ksp.com.intellij.psi.compiled.ClassFileDecompilers
import ksp.org.jetbrains.annotations.TestOnly
import ksp.org.jetbrains.kotlin.analysis.decompiler.psi.KotlinDecompiledFileViewProvider
import ksp.org.jetbrains.kotlin.analysis.decompiler.psi.text.DecompiledText
import ksp.org.jetbrains.kotlin.analysis.decompiler.psi.text.createIncompatibleMetadataVersionDecompiledText
import ksp.org.jetbrains.kotlin.metadata.deserialization.BinaryVersion
import ksp.org.jetbrains.kotlin.metadata.deserialization.MetadataVersion
import ksp.org.jetbrains.kotlin.serialization.SerializerExtensionProtocol
import ksp.org.jetbrains.kotlin.serialization.deserialization.FlexibleTypeDeserializer
import java.io.IOException

abstract class KlibMetadataDecompiler<out V : BinaryVersion>(
    private val fileType: FileType,
    private val serializerProtocol: () -> SerializerExtensionProtocol,
    private val flexibleTypeDeserializer: FlexibleTypeDeserializer,
) : ClassFileDecompilers.Full() {
    protected abstract val metadataStubBuilder: KlibMetadataStubBuilder

    protected abstract fun doReadFile(file: VirtualFile): FileWithMetadata?

    protected abstract fun getDecompiledText(
        fileWithMetadata: FileWithMetadata.Compatible,
        virtualFile: VirtualFile,
        serializerProtocol: SerializerExtensionProtocol,
        flexibleTypeDeserializer: FlexibleTypeDeserializer
    ): DecompiledText

    override fun accepts(file: VirtualFile) = FileTypeRegistry.getInstance().isFileOfType(file, fileType)

    override fun getStubBuilder() = metadataStubBuilder

    override fun createFileViewProvider(file: VirtualFile, manager: PsiManager, physical: Boolean) =
        KotlinDecompiledFileViewProvider(manager, file, physical) { provider ->
            KlibDecompiledFile(
                provider,
                ::buildDecompiledText
            )
        }

    protected fun readFileSafely(file: VirtualFile): FileWithMetadata? {
        if (!file.isValid) return null

        return try {
            doReadFile(file)
        } catch (e: IOException) {
            // This is needed because sometimes we're given VirtualFile instances that point to non-existent .jar entries.
            // Such files are valid (isValid() returns true), but an attempt to read their contents results in a FileNotFoundException.
            // Note that although calling "refresh()" instead of catching an exception would seem more correct here,
            // it's not always allowed and also is likely to degrade performance
            null
        }
    }

    private fun buildDecompiledText(virtualFile: VirtualFile): DecompiledText {
        assert(FileTypeRegistry.getInstance().isFileOfType(virtualFile, fileType)) { "Unexpected file type ${virtualFile.fileType}" }

        return when (val fileWithMetadata = readFileSafely(virtualFile)) {
            is FileWithMetadata.Incompatible -> {
                createIncompatibleMetadataVersionDecompiledText(MetadataVersion.INSTANCE_NEXT, fileWithMetadata.version)
            }
            is FileWithMetadata.Compatible -> {
                getDecompiledText(fileWithMetadata, virtualFile, serializerProtocol(), flexibleTypeDeserializer)
            }
            null -> {
                createIncompatibleMetadataVersionDecompiledText(MetadataVersion.INSTANCE_NEXT, MetadataVersion.INVALID_VERSION)
            }
        }
    }

    @TestOnly
    internal fun buildDecompiledTextForTests(virtualFile: VirtualFile): DecompiledText = buildDecompiledText(virtualFile)
}