/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.symbols.pointers

import ksp.org.jetbrains.kotlin.analysis.api.KaImplementationDetail
import ksp.org.jetbrains.kotlin.analysis.api.KaSession
import ksp.org.jetbrains.kotlin.analysis.api.fir.KaFirSession
import ksp.org.jetbrains.kotlin.analysis.api.impl.base.symbols.pointers.KaBaseCachedSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaCallableSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.resolve.providers.symbolProvider
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirCallableSymbol
import ksp.org.jetbrains.kotlin.name.CallableId

internal abstract class KaTopLevelCallableSymbolPointer<S : KaCallableSymbol>(
    private val callableId: CallableId,
    originalSymbol: S?,
) : KaBaseCachedSymbolPointer<S>(originalSymbol) {
    @KaImplementationDetail
    final override fun restoreIfNotCached(analysisSession: KaSession): S? {
        require(analysisSession is KaFirSession)
        val candidates = analysisSession.getCallableSymbols(callableId)
        if (candidates.isEmpty()) return null
        val session = candidates.first().fir.moduleData.session
        return analysisSession.chooseCandidateAndCreateSymbol(candidates, session)
    }

    protected abstract fun KaFirSession.chooseCandidateAndCreateSymbol(
        candidates: Collection<FirCallableSymbol<*>>,
        firSession: FirSession
    ): S?

    abstract override fun pointsToTheSameSymbolAs(other: KaSymbolPointer<KaSymbol>): Boolean
    protected fun hasTheSameOwner(other: KaTopLevelCallableSymbolPointer<*>): Boolean = other.callableId == callableId
}

private fun KaFirSession.getCallableSymbols(callableId: CallableId) =
    firSession.symbolProvider.getTopLevelCallableSymbols(callableId.packageName, callableId.callableName)

