/*
 * Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
 */
package ksp.com.intellij.psi.tree;

import ksp.com.intellij.lang.ASTNode;
import ksp.com.intellij.lang.LanguageExtension;
import ksp.com.intellij.lang.LighterASTNode;
import ksp.com.intellij.psi.PsiFile;
import ksp.com.intellij.util.ThreeState;
import ksp.com.intellij.util.diff.FlyweightCapableTreeStructure;
import ksp.org.jetbrains.annotations.NotNull;

import java.util.List;

/**
 * Provide this custom comparator to detect whether to reparse lazy-reparseable children of the node for the language:
 * - For instance, if parsing of some IReparseableElementType depends on its parent contents, a custom comparator is needed to check this.
 * - Having a custom comparator as an extension point helps for correct comparing also for embedded/injected fragments.
 */
public interface CustomLanguageASTComparator {
  LanguageExtension<CustomLanguageASTComparator> EXTENSION_POINT_NAME = new LanguageExtension<>("com.intellij.tree.CustomLanguageASTComparator");

  static List<CustomLanguageASTComparator> getMatchingComparators(@NotNull PsiFile file) {
    return EXTENSION_POINT_NAME.allForLanguage(file.getLanguage());
  }

  /**
   * @return {@code ThreeState#NO} for the children to be reparsed, {@code ThreeState#UNSURE} to continue comparing
   */
  @NotNull
  ThreeState compareAST(@NotNull ASTNode oldNode,
                        @NotNull LighterASTNode newNode,
                        @NotNull FlyweightCapableTreeStructure<LighterASTNode> structure);
}
