// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.psi.impl.source.resolve.reference;

import ksp.com.intellij.codeInsight.highlighting.PassRunningAssert;
import ksp.com.intellij.lang.Language;
import ksp.com.intellij.openapi.application.AccessToken;
import ksp.com.intellij.openapi.application.ApplicationManager;
import ksp.com.intellij.openapi.progress.ProgressIndicatorProvider;
import ksp.com.intellij.psi.*;
import ksp.com.intellij.psi.util.CachedValueProvider.Result;
import ksp.com.intellij.psi.util.CachedValuesManager;
import ksp.com.intellij.psi.util.PsiModificationTracker;
import ksp.com.intellij.util.ProcessingContext;
import ksp.org.jetbrains.annotations.ApiStatus;
import ksp.org.jetbrains.annotations.NotNull;

public abstract class ReferenceProvidersRegistry {
  private static final PassRunningAssert CONTRIBUTING_REFERENCES = new PassRunningAssert(
    "the expensive method should not be called during the references contributing");

  public static final PsiReferenceProvider NULL_REFERENCE_PROVIDER = new PsiReferenceProvider() {
      @Override
      public PsiReference @NotNull [] getReferencesByElement(@NotNull PsiElement element, @NotNull ProcessingContext context) {
        return PsiReference.EMPTY_ARRAY;
      }
    };

  public static ReferenceProvidersRegistry getInstance() {
    return ApplicationManager.getApplication().getService(ReferenceProvidersRegistry.class);
  }

  public abstract @NotNull PsiReferenceRegistrar getRegistrar(@NotNull Language language);

  public static PsiReference @NotNull [] getReferencesFromProviders(@NotNull PsiElement context) {
    return getReferencesFromProviders(context, PsiReferenceService.Hints.NO_HINTS);
  }

  public static PsiReference @NotNull [] getReferencesFromProviders(@NotNull PsiElement context, @NotNull PsiReferenceService.Hints hints) {
    ProgressIndicatorProvider.checkCanceled();

    if (hints == PsiReferenceService.Hints.NO_HINTS) {
      return CachedValuesManager.getCachedValue(context, () -> {
        try (AccessToken ignored = CONTRIBUTING_REFERENCES.runPass()) {
          return Result.create(getInstance().doGetReferencesFromProviders(context, PsiReferenceService.Hints.NO_HINTS),
                               PsiModificationTracker.MODIFICATION_COUNT);
        }
      }).clone();
    }

    try (AccessToken ignored = CONTRIBUTING_REFERENCES.runPass()) {
      return getInstance().doGetReferencesFromProviders(context, hints);
    }
  }

  public static void assertNotContributingReferences() {
    CONTRIBUTING_REFERENCES.assertPassNotRunning();
  }

  /**
   * @deprecated suppressing this assert could cause a performance flaw
   */
  @ApiStatus.Internal
  @Deprecated
  public static AccessToken suppressAssertNotContributingReferences() {
    return CONTRIBUTING_REFERENCES.suppressAssertInPass();
  }

  @ApiStatus.Internal
  public abstract void unloadProvidersFor(@NotNull Language language);

  protected abstract PsiReference @NotNull [] doGetReferencesFromProviders(@NotNull PsiElement context, @NotNull PsiReferenceService.Hints hints);
}
