/*
 * Copyright 2000-2009 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ksp.com.intellij.psi.impl.source;

import ksp.com.intellij.lang.ASTNode;
import ksp.com.intellij.openapi.diagnostic.Logger;
import ksp.com.intellij.psi.*;
import ksp.com.intellij.psi.impl.PsiImplUtil;
import ksp.com.intellij.psi.impl.java.stubs.JavaStubElementTypes;
import ksp.com.intellij.psi.impl.java.stubs.PsiParameterListStub;
import ksp.com.intellij.psi.impl.source.tree.CompositeElement;
import ksp.com.intellij.psi.stubs.StubElement;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

public class PsiParameterListImpl extends JavaStubPsiElement<PsiParameterListStub> implements PsiParameterList {
  private static final Logger LOG = Logger.getInstance(PsiParameterListImpl.class);

  public PsiParameterListImpl(@NotNull PsiParameterListStub stub) {
    super(stub, JavaStubElementTypes.PARAMETER_LIST);
  }

  public PsiParameterListImpl(@NotNull ASTNode node) {
    super(node);
  }

  @Override
  public PsiParameter @NotNull [] getParameters() {
    return getStubOrPsiChildren(JavaStubElementTypes.PARAMETER, PsiParameter.ARRAY_FACTORY);
  }

  @Override
  public int getParameterIndex(@NotNull PsiParameter parameter) {
    PsiElement parent = parameter.getParent();
    if (parent != this) {
      LOG.error("Not my parameter; parameter class = " + parameter.getClass() + "; " +
                "this class = " + getClass() + "; " +
                "parameter parent class = " + (parent == null ? null : parent.getClass()));
    }
    return PsiImplUtil.getParameterIndex(parameter, this);
  }

  @Override
  @NotNull
  public CompositeElement getNode() {
    return (CompositeElement)super.getNode();
  }

  @Override
  public int getParametersCount() {
    PsiParameterListStub stub = getGreenStub();
    if (stub != null) {
      int count = 0;
      for (StubElement<?> child : stub.getChildrenStubs()) {
        if (child.getStubType() == JavaStubElementTypes.PARAMETER) {
          count++;
        }
      }
      return count;
    }

    return getNode().countChildren(Constants.PARAMETER_BIT_SET);
  }

  @Nullable
  @Override
  public PsiParameter getParameter(int index) {
    if (index < 0) {
      throw new IllegalArgumentException("index is negative: " + index);
    }
    PsiParameterListStub stub = getGreenStub();
    if (stub != null) {
      int count = 0;
      for (StubElement<?> child : stub.getChildrenStubs()) {
        if (child.getStubType() == JavaStubElementTypes.PARAMETER) {
          if (count == index) return (PsiParameter)child.getPsi(); 
          count++;
        }
      }
    } else {
      CompositeElement node = getNode();
      int count = 0;
      for (ASTNode child = node.getFirstChildNode(); child != null; child = child.getTreeNext()) {
        if (child.getElementType() == JavaStubElementTypes.PARAMETER) {
          if (count == index) return (PsiParameter)child.getPsi();
          count++;
        }
      }
    }
    return null;
  }

  @Override
  public boolean isEmpty() {
    PsiParameterListStub stub = getGreenStub();
    if (stub != null) {
      for (StubElement<?> child : stub.getChildrenStubs()) {
        if (child.getStubType() == JavaStubElementTypes.PARAMETER) {
          return false;
        }
      }
      return true;
    }

    return getNode().findChildByType(Constants.PARAMETER_BIT_SET) == null;
  }

  @Override
  public void accept(@NotNull PsiElementVisitor visitor){
    if (visitor instanceof JavaElementVisitor) {
      ((JavaElementVisitor)visitor).visitParameterList(this);
    }
    else {
      visitor.visitElement(this);
    }
  }

  @Override
  @NonNls
  public String toString(){
    return "PsiParameterList:" + getText();
  }
}