// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package ksp.com.intellij.openapi.vfs.encoding;

import ksp.com.intellij.openapi.extensions.ExtensionPointName;
import ksp.com.intellij.openapi.project.Project;
import ksp.com.intellij.openapi.vfs.VirtualFile;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import java.nio.charset.Charset;

/**
 * Implement this interface to override encoding specified in {@link EncodingRegistry} for an arbitrary virtual file
 * and define an extension in {@code plugin.xml}, for example:
 * <pre><code>
 *   &lt;fileEncodingProvider implementation="com.acme.example.MyFileEncodingProvider"/&gt;
 * </code></pre>
 * <b>Note:</b> The provider doesn't affect files defining their own encoding via {@code LanguageFileType.getCharset()}.
 */
public interface FileEncodingProvider {
  ExtensionPointName<FileEncodingProvider> EP_NAME = new ExtensionPointName<>("com.intellij.fileEncodingProvider");

  /**
   * @param virtualFile The virtual file to override encoding for.
   * @param project The context project.
   * @return The encoding to be used for the given virtual file. <b>It should not depend on the current project. Otherwise it may
   * cause index inconsistencies.</b>
   */
  @Nullable
  Charset getEncoding(@NotNull VirtualFile virtualFile, Project project);
}
