// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package ksp.com.intellij.lang.java.parser;

import ksp.com.intellij.core.JavaPsiBundle;
import ksp.com.intellij.lang.ASTNode;
import ksp.com.intellij.lang.LighterLazyParseableNode;
import ksp.com.intellij.lang.PsiBuilder;
import ksp.com.intellij.lang.WhitespacesAndCommentsBinder;
import ksp.com.intellij.lang.impl.TokenSequence;
import ksp.com.intellij.lang.java.JavaParserDefinition;
import ksp.com.intellij.lang.java.lexer.BasicJavaLexer;
import ksp.com.intellij.lang.java.lexer.JavaDocLexer;
import ksp.com.intellij.lexer.TokenList;
import ksp.com.intellij.openapi.util.NlsContexts;
import ksp.com.intellij.openapi.util.Pair;
import ksp.com.intellij.pom.java.LanguageLevel;
import ksp.com.intellij.psi.PsiFile;
import ksp.com.intellij.psi.impl.source.WhiteSpaceAndCommentSetHolder;
import ksp.com.intellij.psi.impl.source.tree.ElementType;
import ksp.com.intellij.psi.tree.IElementType;
import ksp.com.intellij.psi.tree.TokenSet;
import ksp.com.intellij.psi.util.CachedValueProvider;
import ksp.com.intellij.psi.util.CachedValuesManager;
import ksp.com.intellij.psi.util.PsiUtil;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;
import ksp.org.jetbrains.annotations.PropertyKey;

import java.util.function.Predicate;

public final class JavaParserUtil {
  public static final TokenSet WS_COMMENTS = TokenSet.orSet(ElementType.JAVA_COMMENT_BIT_SET, TokenSet.WHITE_SPACE);

  @NotNull
  public static TokenList obtainTokens(@NotNull PsiFile file) {
    return CachedValuesManager.getCachedValue(file, () ->
      CachedValueProvider.Result.create(
        TokenSequence.performLexing(file.getViewProvider().getContents(), JavaParserDefinition.createLexer(PsiUtil.getLanguageLevel(file))),
        file));
  }

  @FunctionalInterface
  public interface ParserWrapper extends BasicJavaParserUtil.ParserWrapper {
  }

  public static final WhitespacesAndCommentsBinder PRECEDING_COMMENT_BINDER =
    WhiteSpaceAndCommentSetHolder.INSTANCE.getPrecedingCommentBinder();
  public static final WhitespacesAndCommentsBinder SPECIAL_PRECEDING_COMMENT_BINDER =
    WhiteSpaceAndCommentSetHolder.INSTANCE.getSpecialPrecedingCommentBinder();
  public static final WhitespacesAndCommentsBinder TRAILING_COMMENT_BINDER =
    WhiteSpaceAndCommentSetHolder.INSTANCE.getTrailingCommentBinder();


  private JavaParserUtil() { }

  public static void setLanguageLevel(final PsiBuilder builder, final LanguageLevel level) {
    BasicJavaParserUtil.setLanguageLevel(builder, level);
  }

  @NotNull
  public static LanguageLevel getLanguageLevel(final PsiBuilder builder) {
    return BasicJavaParserUtil.getLanguageLevel(builder);
  }

  public static void setParseStatementCodeBlocksDeep(final PsiBuilder builder, final boolean deep) {
    BasicJavaParserUtil.setParseStatementCodeBlocksDeep(builder, deep);
  }

  public static boolean isParseStatementCodeBlocksDeep(final PsiBuilder builder) {
    return BasicJavaParserUtil.isParseStatementCodeBlocksDeep(builder);
  }

  @NotNull
  public static PsiBuilder createBuilder(final ASTNode chameleon) {
    return BasicJavaParserUtil.createBuilder(chameleon,
                                             (psi) -> PsiUtil.getLanguageLevel(psi),
                                             (level) -> (BasicJavaLexer)JavaParserDefinition.createLexer(level),
                                             (psi) -> obtainTokens(psi));
  }

  @NotNull
  public static PsiBuilder createBuilder(final LighterLazyParseableNode chameleon) {
    return BasicJavaParserUtil.createBuilder(chameleon,
                                             (psi) -> PsiUtil.getLanguageLevel(psi),
                                             (level) -> (BasicJavaLexer)JavaParserDefinition.createLexer(level));
  }

  @Nullable
  public static ASTNode parseFragment(final ASTNode chameleon, final ParserWrapper wrapper) {
    return BasicJavaParserUtil.parseFragment(chameleon, wrapper,
                                             (level) -> (JavaDocLexer)JavaParserDefinition.createDocLexer(level),
                                             (level) -> (BasicJavaLexer)JavaParserDefinition.createLexer(level)
    );
  }

  @Nullable
  public static ASTNode parseFragment(final ASTNode chameleon,
                                      final BasicJavaParserUtil.ParserWrapper wrapper,
                                      final boolean eatAll,
                                      final LanguageLevel level) {
    return BasicJavaParserUtil.parseFragment(chameleon, wrapper, eatAll, level,
                                             (levelLanguage) -> (JavaDocLexer)JavaParserDefinition.createDocLexer(levelLanguage),
                                             (levelLanguage) -> (BasicJavaLexer)JavaParserDefinition.createLexer(levelLanguage)
    );
  }

  public static void done(final PsiBuilder.Marker marker, final IElementType type) {
    BasicJavaParserUtil.done(marker, type, WhiteSpaceAndCommentSetHolder.INSTANCE);
  }

  @Nullable
  public static IElementType exprType(@Nullable final PsiBuilder.Marker marker) {
    return BasicJavaParserUtil.exprType(marker);
  }

  // used instead of PsiBuilder.error() as it keeps all subsequent error messages
  public static void error(final PsiBuilder builder, @NotNull @NlsContexts.ParsingError String message) {
    BasicJavaParserUtil.error(builder, message);
  }

  public static void error(final PsiBuilder builder,
                           @NotNull @NlsContexts.ParsingError String message,
                           @Nullable final PsiBuilder.Marker before) {
    BasicJavaParserUtil.error(builder, message, before);
  }

  public static boolean expectOrError(PsiBuilder builder,
                                      TokenSet expected,
                                      @PropertyKey(resourceBundle = JavaPsiBundle.BUNDLE) String key) {
    return BasicJavaParserUtil.expectOrError(builder, expected, key);
  }

  public static boolean expectOrError(PsiBuilder builder,
                                      IElementType expected,
                                      @PropertyKey(resourceBundle = JavaPsiBundle.BUNDLE) String key) {
    return BasicJavaParserUtil.expectOrError(builder, expected, key);
  }

  public static void emptyElement(final PsiBuilder builder, final IElementType type) {
    BasicJavaParserUtil.emptyElement(builder, type);
  }

  public static void emptyElement(final PsiBuilder.Marker before, final IElementType type) {
    BasicJavaParserUtil.emptyElement(before, type);
  }

  public static void semicolon(final PsiBuilder builder) {
    BasicJavaParserUtil.semicolon(builder);
  }

  public static PsiBuilder braceMatchingBuilder(final PsiBuilder builder) {
    return BasicJavaParserUtil.braceMatchingBuilder(builder);
  }

  public static PsiBuilder stoppingBuilder(final PsiBuilder builder, final int stopAt) {
    return BasicJavaParserUtil.stoppingBuilder(builder, stopAt);
  }

  public static PsiBuilder stoppingBuilder(final PsiBuilder builder, final Predicate<? super Pair<IElementType, String>> condition) {
    return BasicJavaParserUtil.stoppingBuilder(builder, condition);
  }
}