// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.codeInsight.documentation;

import ksp.com.intellij.openapi.util.text.StringUtil;
import ksp.com.intellij.openapi.vfs.VirtualFile;
import ksp.com.intellij.openapi.vfs.VirtualFileManager;
import ksp.com.intellij.openapi.vfs.ex.http.HttpFileSystem;
import ksp.com.intellij.util.SmartList;
import ksp.org.jetbrains.annotations.Contract;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.regex.Pattern;

public final class PlatformDocumentationUtil {
  private static final @NonNls Pattern ourLtFixupPattern = Pattern.compile("<(?=[^/!\\p{Alpha}])");
  private static final @NonNls String LT_ENTITY = "&lt;";

  public static @Nullable List<String> getHttpRoots(String @NotNull [] roots, String relPath) {
    List<String> result = new SmartList<>();
    for (String root : roots) {
      VirtualFile virtualFile = VirtualFileManager.getInstance().findFileByUrl(root);
      if (virtualFile != null) {
        String url = getDocUrl(virtualFile, relPath);
        if (url != null) result.add(url);
      }
    }

    return result.isEmpty() ? null : result;
  }

  public static @Nullable String getDocUrl(@NotNull VirtualFile root, String relPath) {
    if (root.getFileSystem() instanceof HttpFileSystem) {
      @NonNls String url = StringUtil.trimEnd(root.getUrl(), "/index.html", true);
      if (!url.endsWith("/")) {
        url += "/";
      }
      return url + relPath;
    }
    else {
      VirtualFile file = root.findFileByRelativePath(relPath);
      return file == null ? null : file.getUrl();
    }
  }

  /**
   * Updates HTML contents for display in JEditorPane, which treats invalid HTML somewhat differently than popular browsers.
   */
  @Contract(pure = true)
  public static String fixupText(@NotNull CharSequence docText) {
    return ourLtFixupPattern.matcher(docText).replaceAll(LT_ENTITY);
  }
}
