// Copyright 2000-2022 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.codeInsight;

import ksp.com.intellij.lang.Language;
import ksp.com.intellij.openapi.application.ApplicationManager;
import ksp.com.intellij.psi.PsiClass;
import ksp.com.intellij.psi.PsiElement;
import ksp.com.intellij.psi.PsiMethod;
import ksp.com.intellij.psi.PsiModifierListOwner;
import ksp.com.intellij.psi.util.CachedValueProvider;
import ksp.com.intellij.psi.util.CachedValuesManager;
import ksp.com.intellij.psi.util.PsiModificationTracker;
import ksp.com.intellij.testIntegration.TestFramework;
import ksp.com.intellij.util.containers.ContainerUtil;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import java.util.*;

public abstract class TestFrameworks {
  public static TestFrameworks getInstance() {
    return ApplicationManager.getApplication().getService(TestFrameworks.class);
  }

  public abstract boolean isTestClass(@NotNull PsiClass psiClass);
  public abstract boolean isPotentialTestClass(@NotNull PsiClass psiClass);

  @Nullable
  public abstract PsiMethod findOrCreateSetUpMethod(PsiClass psiClass);

  @Nullable
  public abstract PsiMethod findSetUpMethod(PsiClass psiClass);

  @Nullable
  public abstract PsiMethod findTearDownMethod(PsiClass psiClass);

  protected abstract boolean hasConfigMethods(PsiClass psiClass);

  public abstract boolean isTestMethod(PsiMethod method);

  /**
   * Checks method on the possibility to run as a test
   *
   * @param method        method element to check
   * @param checkAbstract the fact that an abstract class is a test or not, if false then is test
   * @return the result of checking
   */
  public boolean isTestMethod(PsiMethod method, boolean checkAbstract) {
    return isTestMethod(method);
  }

  public boolean isTestOrConfig(PsiClass psiClass) {
    return isTestClass(psiClass) || hasConfigMethods(psiClass);
  }

  @Nullable
  public static TestFramework detectFramework(@NotNull final PsiClass psiClass) {
    return ContainerUtil.getFirstItem(detectApplicableFrameworks(psiClass));
  }

  @NotNull
  public static Set<TestFramework> detectApplicableFrameworks(@NotNull final PsiClass psiClass) {
    PsiModifierListOwner normalized = AnnotationCacheOwnerNormalizer.normalize(psiClass);
    return CachedValuesManager.getCachedValue(normalized, () -> CachedValueProvider.Result
      .create(computeFrameworks(normalized), PsiModificationTracker.MODIFICATION_COUNT));
  }

  private static Set<TestFramework> computeFrameworks(PsiElement psiClass) {
    Set<TestFramework> frameworks = new LinkedHashSet<>();

    Language classLanguage = psiClass.getLanguage();
    Map<String, Language> checkedFrameworksByName = new HashMap<>();

    for (TestFramework framework : TestFramework.EXTENSION_NAME.getExtensionList()) {
      String frameworkName = framework.getName();
      Language frameworkLanguage = framework.getLanguage();

      Language checkedFrameworkLanguage = checkedFrameworksByName.get(frameworkName);
      // if we've checked framework for more specific language - no reasons to check it again for more general language
      if (checkedFrameworkLanguage != null && isSubLanguage(checkedFrameworkLanguage, frameworkLanguage)) continue;

      if (!isSubLanguage(classLanguage, frameworkLanguage))
        continue;

      if (framework.isTestClass(psiClass) ||
          framework.findSetUpMethod(psiClass) != null ||
          framework.findTearDownMethod(psiClass) != null) {
        frameworks.add(framework);
      }
      checkedFrameworksByName.put(frameworkName, frameworkLanguage);
    }
    return frameworks;
  }

  /**
   * @return <code>true</code> if <code>framework</code> could handle element by its language
   */
  public static boolean isSuitableByLanguage(PsiElement element, TestFramework framework) {
    return element.getContainingFile() != null && isSubLanguage(element.getLanguage(), framework.getLanguage());
  }

  private static boolean isSubLanguage(@NotNull Language language, @NotNull Language parentLanguage) {
    return parentLanguage == Language.ANY || language.isKindOf(parentLanguage);
  }
}
