/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.storage.contrib.nio;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.NonWritableChannelException;
import java.nio.channels.SeekableByteChannel;
import java.util.ArrayList;
import java.util.List;
import java.util.UnknownFormatConversionException;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

public final class SeekableByteChannelPrefetcher
implements SeekableByteChannel {
    private final SeekableByteChannel chan;
    private final int bufSize;
    private final ExecutorService exec;
    private final long size;
    private final List<WorkUnit> full = new ArrayList<WorkUnit>();
    private WorkUnit fetching;
    private static final int BUF_COUNT = 2;
    private long position;
    private boolean open;
    private Stopwatch betweenCallsToRead = Stopwatch.createUnstarted();
    private static int prefetcherCount;
    private long msWaitingForData;
    private long msCopyingData;
    private long bytesReturned;
    private long bytesRead;
    private long msBetweenCallsToRead;
    private long nbHit;
    private long nbNearHit;
    private long nbMiss;
    private long nbGoingBack;
    private long nbReadsPastEnd;
    private static final boolean trackTime = false;

    public static SeekableByteChannel addPrefetcher(int bufferSizeMB, SeekableByteChannel channel) throws IOException {
        return new SeekableByteChannelPrefetcher(channel, bufferSizeMB * 1024 * 1024);
    }

    private SeekableByteChannelPrefetcher(SeekableByteChannel chan, int bufSize) throws IOException {
        Preconditions.checkArgument((!(chan instanceof SeekableByteChannelPrefetcher) ? 1 : 0) != 0, (Object)"Cannot wrap a prefetcher with a prefetcher.");
        if (!chan.isOpen()) {
            throw new IllegalArgumentException("channel must be open");
        }
        this.chan = chan;
        if (bufSize <= 0) {
            throw new IllegalArgumentException("bufSize must be positive");
        }
        this.size = chan.size();
        this.bufSize = (long)bufSize > this.size ? (int)this.size : bufSize;
        this.open = true;
        int prefetcherIndex = prefetcherCount++;
        String nameFormat = "nio-prefetcher-" + prefetcherIndex + "-thread-%d";
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat(nameFormat).setDaemon(true).build();
        this.exec = Executors.newFixedThreadPool(1, threadFactory);
    }

    public Statistics getStatistics() {
        return new Statistics(this.msWaitingForData, this.msCopyingData, this.bytesReturned, this.bytesRead, this.msBetweenCallsToRead, this.nbHit, this.nbNearHit, this.nbMiss, this.nbGoingBack, this.nbReadsPastEnd);
    }

    private void ensureFetching(long blockIndex) {
        if (this.fetching != null) {
            if (this.fetching.futureBuf.isDone()) {
                this.full.add(this.fetching);
                this.fetching = null;
            } else {
                return;
            }
        }
        for (WorkUnit w : this.full) {
            if (w.blockIndex != blockIndex) continue;
            return;
        }
        if (this.full.size() < 2) {
            this.fetching = new WorkUnit(this.chan, this.bufSize, blockIndex);
        } else {
            this.fetching = this.full.remove(0);
            this.fetching.resetForIndex(blockIndex);
        }
        this.bytesRead += (long)this.bufSize;
        this.fetching.futureBuf = this.exec.submit(this.fetching);
    }

    public ByteBuffer fetch(long position) throws InterruptedException, ExecutionException {
        long blockIndex = position / (long)this.bufSize;
        boolean goingBack = false;
        for (WorkUnit w : this.full) {
            if (w.blockIndex == blockIndex) {
                this.ensureFetching(blockIndex + 1L);
                ++this.nbHit;
                return w.buf;
            }
            if (w.blockIndex <= blockIndex) continue;
            goingBack = true;
        }
        if (goingBack) {
            ++this.nbGoingBack;
        }
        if (null == this.fetching) {
            this.ensureFetching(blockIndex);
        }
        WorkUnit candidate = this.fetching;
        ByteBuffer buf = candidate.getBuf();
        this.full.add(candidate);
        this.fetching = null;
        if (candidate.blockIndex == blockIndex) {
            ++this.nbNearHit;
            this.ensureFetching(blockIndex + 1L);
            return buf;
        }
        ++this.nbMiss;
        this.ensureFetching(blockIndex);
        candidate = this.fetching;
        buf = candidate.getBuf();
        this.full.add(candidate);
        this.fetching = null;
        this.ensureFetching(blockIndex + 1L);
        return buf;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public synchronized int read(ByteBuffer dst) throws IOException {
        ByteBuffer src;
        if (!this.open) {
            throw new ClosedChannelException();
        }
        try {
            src = this.fetch(this.position);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return 0;
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
        if (null == src) {
            ++this.nbReadsPastEnd;
            return -1;
        }
        long blockIndex = this.position / (long)this.bufSize;
        int offset = (int)(this.position - blockIndex * (long)this.bufSize);
        int availableToCopy = src.position() - offset;
        if (availableToCopy < 0) {
            ++this.nbReadsPastEnd;
            return -1;
        }
        int bytesToCopy = dst.remaining();
        byte[] array = src.array();
        if (availableToCopy < bytesToCopy) {
            bytesToCopy = availableToCopy;
        }
        dst.put(array, offset, bytesToCopy);
        this.position += (long)bytesToCopy;
        this.bytesReturned += (long)bytesToCopy;
        if (availableToCopy != 0) return bytesToCopy;
        return -1;
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        throw new NonWritableChannelException();
    }

    @Override
    public long position() throws IOException {
        if (!this.open) {
            throw new ClosedChannelException();
        }
        return this.position;
    }

    @Override
    public SeekableByteChannel position(long newPosition) throws IOException {
        if (!this.open) {
            throw new ClosedChannelException();
        }
        this.position = newPosition;
        return this;
    }

    @Override
    public long size() throws IOException {
        if (!this.open) {
            throw new ClosedChannelException();
        }
        return this.size;
    }

    @Override
    public SeekableByteChannel truncate(long size) throws IOException {
        throw new NonWritableChannelException();
    }

    @Override
    public boolean isOpen() {
        return this.open;
    }

    @Override
    public void close() throws IOException {
        if (this.open) {
            this.exec.shutdownNow();
            try {
                this.exec.awaitTermination(1L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            this.chan.close();
            this.open = false;
        }
    }

    private static class WorkUnit
    implements Callable<ByteBuffer>,
    Closeable {
        public final ByteBuffer buf;
        public long blockIndex;
        private final SeekableByteChannel chan;
        private final int blockSize;
        private Future<ByteBuffer> futureBuf;

        public WorkUnit(SeekableByteChannel chan, int blockSize, long blockIndex) {
            this.chan = chan;
            this.buf = ByteBuffer.allocate(blockSize);
            this.futureBuf = null;
            this.blockSize = blockSize;
            this.blockIndex = blockIndex;
        }

        @Override
        public ByteBuffer call() throws IOException {
            long pos = (long)this.blockSize * this.blockIndex;
            if (pos > this.chan.size()) {
                return null;
            }
            if (pos < 0L) {
                throw new IllegalArgumentException("blockIndex " + this.blockIndex + " has position " + pos + ": negative position is not valid.");
            }
            this.chan.position(pos);
            while (this.chan.read(this.buf) >= 0 && this.buf.hasRemaining()) {
            }
            return this.buf;
        }

        public ByteBuffer getBuf() throws ExecutionException, InterruptedException {
            return this.futureBuf.get();
        }

        public WorkUnit resetForIndex(long blockIndex) {
            this.blockIndex = blockIndex;
            this.buf.clear();
            this.futureBuf = null;
            return this;
        }

        @Override
        public void close() throws IOException {
            this.chan.close();
        }
    }

    public static class Statistics {
        public final long msWaitingForData;
        public final long msCopyingData;
        public final long bytesReturned;
        public final long bytesRead;
        public final long msBetweenCallsToRead;
        public final long nbHit;
        public final long nbNearHit;
        public final long nbMiss;
        public final long nbGoingBack;
        public final long nbReadsPastEnd;

        private Statistics(long msWaitingForData, long msCopyingData, long bytesReturned, long bytesRead, long msBetweenCallsToRead, long nbHit, long nbNearHit, long nbMiss, long nbGoingBack, long nbReadsPastEnd) {
            this.msWaitingForData = msWaitingForData;
            this.msCopyingData = msCopyingData;
            this.bytesReturned = bytesReturned;
            this.bytesRead = bytesRead;
            this.msBetweenCallsToRead = msBetweenCallsToRead;
            this.nbHit = nbHit;
            this.nbNearHit = nbNearHit;
            this.nbMiss = nbMiss;
            this.nbGoingBack = nbGoingBack;
            this.nbReadsPastEnd = nbReadsPastEnd;
        }

        public String toString() {
            try {
                double returnedPct = this.bytesRead > 0L ? 100.0 * (double)this.bytesReturned / (double)this.bytesRead : 100.0;
                return String.format("Bytes read: %12d\n  returned: %12d ( %3.2f %% )", this.bytesRead, this.bytesReturned, returnedPct) + String.format("\nReads past the end: %3d", this.nbReadsPastEnd) + String.format("\nReads forcing re-fetching of an earlier block: %3d", this.nbGoingBack) + String.format("\nCache\n hits:      %12d\n near-hits: %12d\n misses:    %12d", this.nbHit, this.nbNearHit, this.nbMiss);
            }
            catch (UnknownFormatConversionException x) {
                return "(error while formatting statistics)";
            }
        }
    }
}

