/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.firestore;

import com.google.api.core.ApiAsyncFunction;
import com.google.api.core.ApiFuture;
import com.google.api.core.ApiFutures;
import com.google.api.core.CurrentMillisClock;
import com.google.api.core.SettableApiFuture;
import com.google.api.gax.retrying.ExponentialRetryAlgorithm;
import com.google.api.gax.retrying.TimedAttemptSettings;
import com.google.cloud.firestore.BatchWriteResult;
import com.google.cloud.firestore.BulkCommitBatch;
import com.google.cloud.firestore.DocumentReference;
import com.google.cloud.firestore.FieldPath;
import com.google.cloud.firestore.FirestoreImpl;
import com.google.cloud.firestore.Precondition;
import com.google.cloud.firestore.RateLimiter;
import com.google.cloud.firestore.SetOptions;
import com.google.cloud.firestore.UpdateBuilder;
import com.google.cloud.firestore.WriteResult;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.MoreExecutors;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

final class BulkWriter
implements AutoCloseable {
    public static final int MAX_BATCH_SIZE = 500;
    public static final int MAX_RETRY_ATTEMPTS = 10;
    private static final int STARTING_MAXIMUM_OPS_PER_SECOND = 500;
    private static final double RATE_LIMITER_MULTIPLIER = 1.5;
    private static final int RATE_LIMITER_MULTIPLIER_MILLIS = 300000;
    private static final Logger logger = Logger.getLogger(BulkWriter.class.getName());
    private int maxBatchSize = 500;
    private final List<BulkCommitBatch> batchQueue = new CopyOnWriteArrayList<BulkCommitBatch>();
    private boolean closed = false;
    private final RateLimiter rateLimiter;
    private final FirestoreImpl firestore;
    private final ScheduledExecutorService firestoreExecutor;
    private final ExponentialRetryAlgorithm backoff;
    private TimedAttemptSettings nextAttempt;

    BulkWriter(FirestoreImpl firestore, boolean enableThrottling) {
        this.firestore = firestore;
        this.backoff = new ExponentialRetryAlgorithm(firestore.getOptions().getRetrySettings(), CurrentMillisClock.getDefaultClock());
        this.nextAttempt = this.backoff.createFirstAttempt();
        this.firestoreExecutor = firestore.getClient().getExecutor();
        this.rateLimiter = enableThrottling ? new RateLimiter(500, 1.5, 300000) : new RateLimiter(Integer.MAX_VALUE, Double.MAX_VALUE, Integer.MAX_VALUE);
    }

    @Nonnull
    public ApiFuture<WriteResult> create(@Nonnull DocumentReference documentReference, @Nonnull Map<String, Object> fields) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.create(documentReference, fields);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> create(@Nonnull DocumentReference documentReference, @Nonnull Object pojo) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.create(documentReference, pojo);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> delete(@Nonnull DocumentReference documentReference) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.delete(documentReference);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> delete(@Nonnull DocumentReference documentReference, @Nonnull Precondition precondition) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.delete(documentReference, precondition);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> set(@Nonnull DocumentReference documentReference, @Nonnull Map<String, Object> fields) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.set(documentReference, fields);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> set(@Nonnull DocumentReference documentReference, @Nonnull Map<String, Object> fields, @Nonnull SetOptions options) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.set(documentReference, fields, options);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> set(@Nonnull DocumentReference documentReference, Object pojo, @Nonnull SetOptions options) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.set(documentReference, pojo, options);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> set(@Nonnull DocumentReference documentReference, Object pojo) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.set(documentReference, pojo);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> update(@Nonnull DocumentReference documentReference, @Nonnull Map<String, Object> fields) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.update(documentReference, fields);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> update(@Nonnull DocumentReference documentReference, @Nonnull Map<String, Object> fields, Precondition precondition) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.update(documentReference, fields, precondition);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> update(@Nonnull DocumentReference documentReference, @Nonnull String field, @Nullable Object value, Object ... moreFieldsAndValues) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.update(documentReference, field, value, moreFieldsAndValues);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> update(@Nonnull DocumentReference documentReference, @Nonnull FieldPath fieldPath, @Nullable Object value, Object ... moreFieldsAndValues) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.update(documentReference, fieldPath, value, moreFieldsAndValues);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> update(@Nonnull DocumentReference documentReference, @Nonnull Precondition precondition, @Nonnull String field, @Nullable Object value, Object ... moreFieldsAndValues) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.update(documentReference, precondition, field, value, moreFieldsAndValues);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<WriteResult> update(@Nonnull DocumentReference documentReference, @Nonnull Precondition precondition, @Nonnull FieldPath fieldPath, @Nullable Object value, Object ... moreFieldsAndValues) {
        this.verifyNotClosed();
        BulkCommitBatch bulkCommitBatch = this.getEligibleBatch(documentReference);
        ApiFuture future = (ApiFuture)bulkCommitBatch.update(documentReference, precondition, fieldPath, value, moreFieldsAndValues);
        this.sendReadyBatches();
        return future;
    }

    @Nonnull
    public ApiFuture<Void> flush() {
        this.verifyNotClosed();
        final SettableApiFuture flushComplete = SettableApiFuture.create();
        ArrayList<SettableApiFuture<WriteResult>> writeFutures = new ArrayList<SettableApiFuture<WriteResult>>();
        for (BulkCommitBatch batch : this.batchQueue) {
            batch.markReadyToSend();
            writeFutures.addAll(batch.getPendingFutures());
        }
        this.sendReadyBatches();
        ApiFutures.successfulAsList(writeFutures).addListener(new Runnable(){

            @Override
            public void run() {
                flushComplete.set(null);
            }
        }, MoreExecutors.directExecutor());
        return flushComplete;
    }

    @Override
    public void close() throws InterruptedException, ExecutionException {
        ApiFuture<Void> flushFuture = this.flush();
        this.closed = true;
        flushFuture.get();
    }

    private void verifyNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("BulkWriter has already been closed.");
        }
    }

    private BulkCommitBatch getEligibleBatch(DocumentReference documentReference) {
        BulkCommitBatch lastBatch;
        if (this.batchQueue.size() > 0 && (lastBatch = this.batchQueue.get(this.batchQueue.size() - 1)).getState() == UpdateBuilder.BatchState.OPEN && !lastBatch.hasDocument(documentReference)) {
            return lastBatch;
        }
        return this.createNewBatch();
    }

    private BulkCommitBatch createNewBatch() {
        BulkCommitBatch newBatch = new BulkCommitBatch(this.firestore, this.maxBatchSize);
        if (this.batchQueue.size() > 0) {
            this.batchQueue.get(this.batchQueue.size() - 1).markReadyToSend();
            this.sendReadyBatches();
        }
        this.batchQueue.add(newBatch);
        return newBatch;
    }

    private void sendReadyBatches() {
        ImmutableList unsentBatches = FluentIterable.from(this.batchQueue).filter((Predicate)new Predicate<BulkCommitBatch>(){

            public boolean apply(BulkCommitBatch batch) {
                return batch.getState() == UpdateBuilder.BatchState.READY_TO_SEND;
            }
        }).toList();
        for (int index = 0; index < unsentBatches.size() && this.isBatchSendable((BulkCommitBatch)unsentBatches.get(index)); ++index) {
            final BulkCommitBatch batch = (BulkCommitBatch)unsentBatches.get(index);
            long delayMs = this.rateLimiter.getNextRequestDelayMs(batch.getPendingOperationCount());
            Preconditions.checkState((delayMs != -1L ? 1 : 0) != 0, (Object)"Batch size should be under capacity");
            if (delayMs != 0L) {
                this.firestoreExecutor.schedule(new Runnable(){

                    @Override
                    public void run() {
                        BulkWriter.this.sendBatch(batch);
                    }
                }, delayMs, TimeUnit.MILLISECONDS);
                break;
            }
            this.sendBatch(batch);
        }
    }

    private void sendBatch(final BulkCommitBatch batch) {
        Preconditions.checkState((batch.state == UpdateBuilder.BatchState.READY_TO_SEND ? 1 : 0) != 0, (Object)"The batch should be marked as READY_TO_SEND before committing");
        batch.state = UpdateBuilder.BatchState.SENT;
        boolean success = this.rateLimiter.tryMakeRequest(batch.getPendingOperationCount());
        Preconditions.checkState((boolean)success, (Object)"Batch should be under rate limit to be sent.");
        ApiFuture<Void> commitFuture = this.bulkCommit(batch);
        commitFuture.addListener(new Runnable(){

            @Override
            public void run() {
                boolean removed = BulkWriter.this.batchQueue.remove(batch);
                Preconditions.checkState((boolean)removed, (Object)("The batch should be in the BatchQueue." + BulkWriter.this.batchQueue.size()));
                BulkWriter.this.sendReadyBatches();
            }
        }, MoreExecutors.directExecutor());
    }

    private ApiFuture<Void> bulkCommit(BulkCommitBatch batch) {
        return this.bulkCommit(batch, 0);
    }

    private ApiFuture<Void> bulkCommit(BulkCommitBatch batch, int attempt) {
        final SettableApiFuture backoffFuture = SettableApiFuture.create();
        this.firestoreExecutor.schedule(new Runnable(){

            @Override
            public void run() {
                backoffFuture.set(null);
            }
        }, this.nextAttempt.getRandomizedRetryDelay().toMillis(), TimeUnit.MILLISECONDS);
        return ApiFutures.transformAsync((ApiFuture)backoffFuture, (ApiAsyncFunction)new BackoffCallback(batch, attempt), (Executor)this.firestoreExecutor);
    }

    private boolean isBatchSendable(BulkCommitBatch batch) {
        if (!batch.getState().equals((Object)UpdateBuilder.BatchState.READY_TO_SEND)) {
            return false;
        }
        for (final DocumentReference documentReference : batch.getPendingDocuments()) {
            boolean isRefInFlight = FluentIterable.from(this.batchQueue).anyMatch((Predicate)new Predicate<BulkCommitBatch>(){

                public boolean apply(BulkCommitBatch batch) {
                    return batch.getState().equals((Object)UpdateBuilder.BatchState.SENT) && batch.hasDocument(documentReference);
                }
            });
            if (!isRefInFlight) continue;
            logger.log(Level.WARNING, String.format("Duplicate write to document %s detected. Writing to the same document multiple times will slow down BulkWriter. Write to unique documents in order to maximize throughput.", documentReference.getPath()));
            return false;
        }
        return true;
    }

    @VisibleForTesting
    void setMaxBatchSize(int size) {
        this.maxBatchSize = size;
    }

    private class ProcessBulkCommitCallback
    implements ApiAsyncFunction<List<BatchWriteResult>, Void> {
        final BulkCommitBatch batch;
        final int attempt;

        public ProcessBulkCommitCallback(BulkCommitBatch batch, int attempt) {
            this.batch = batch;
            this.attempt = attempt;
        }

        public ApiFuture<Void> apply(List<BatchWriteResult> results) {
            this.batch.processResults(results);
            Set<DocumentReference> remainingOps = this.batch.getPendingDocuments();
            if (!remainingOps.isEmpty()) {
                logger.log(Level.WARNING, String.format("Current batch failed at retry #%d. Num failures: %d", this.attempt, remainingOps.size()));
                if (this.attempt < 10) {
                    BulkWriter.this.nextAttempt = BulkWriter.this.backoff.createNextAttempt(BulkWriter.this.nextAttempt);
                    BulkCommitBatch newBatch = new BulkCommitBatch(BulkWriter.this.firestore, this.batch, remainingOps);
                    return BulkWriter.this.bulkCommit(newBatch, this.attempt + 1);
                }
                this.batch.failRemainingOperations(results);
            }
            return ApiFutures.immediateFuture(null);
        }
    }

    private class BackoffCallback
    implements ApiAsyncFunction<Void, Void> {
        final BulkCommitBatch batch;
        final int attempt;

        public BackoffCallback(BulkCommitBatch batch, int attempt) {
            this.batch = batch;
            this.attempt = attempt;
        }

        public ApiFuture<Void> apply(Void ignored) {
            return ApiFutures.transformAsync((ApiFuture)ApiFutures.catchingAsync(this.batch.bulkCommit(), Exception.class, (ApiAsyncFunction)new ApiAsyncFunction<Exception, List<BatchWriteResult>>(){

                public ApiFuture<List<BatchWriteResult>> apply(Exception exception) {
                    ArrayList<BatchWriteResult> results = new ArrayList<BatchWriteResult>();
                    for (DocumentReference documentReference : BackoffCallback.this.batch.getPendingDocuments()) {
                        results.add(new BatchWriteResult(documentReference, null, exception));
                    }
                    return ApiFutures.immediateFuture(results);
                }
            }, (Executor)MoreExecutors.directExecutor()), (ApiAsyncFunction)new ProcessBulkCommitCallback(this.batch, this.attempt), (Executor)MoreExecutors.directExecutor());
        }
    }
}

