/*
 * Copyright 2025 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.cloud.dialogflow.cx.v3;

import com.google.api.core.ApiFuture;
import com.google.api.core.ApiFutures;
import com.google.api.gax.core.BackgroundResource;
import com.google.api.gax.paging.AbstractFixedSizeCollection;
import com.google.api.gax.paging.AbstractPage;
import com.google.api.gax.paging.AbstractPagedListResponse;
import com.google.api.gax.rpc.PageContext;
import com.google.api.gax.rpc.UnaryCallable;
import com.google.cloud.dialogflow.cx.v3.stub.SecuritySettingsServiceStub;
import com.google.cloud.dialogflow.cx.v3.stub.SecuritySettingsServiceStubSettings;
import com.google.cloud.location.GetLocationRequest;
import com.google.cloud.location.ListLocationsRequest;
import com.google.cloud.location.ListLocationsResponse;
import com.google.cloud.location.Location;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.protobuf.Empty;
import com.google.protobuf.FieldMask;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;
import javax.annotation.Generated;

// AUTO-GENERATED DOCUMENTATION AND CLASS.
/**
 * Service Description: Service for managing security settings for Dialogflow.
 *
 * <p>This class provides the ability to make remote calls to the backing service through method
 * calls that map to API methods. Sample code to get started:
 *
 * <pre>{@code
 * // This snippet has been automatically generated and should be regarded as a code template only.
 * // It will require modifications to work:
 * // - It may require correct/in-range values for request initialization.
 * // - It may require specifying regional endpoints when creating the service client as shown in
 * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
 * try (SecuritySettingsServiceClient securitySettingsServiceClient =
 *     SecuritySettingsServiceClient.create()) {
 *   LocationName parent = LocationName.of("[PROJECT]", "[LOCATION]");
 *   SecuritySettings securitySettings = SecuritySettings.newBuilder().build();
 *   SecuritySettings response =
 *       securitySettingsServiceClient.createSecuritySettings(parent, securitySettings);
 * }
 * }</pre>
 *
 * <p>Note: close() needs to be called on the SecuritySettingsServiceClient object to clean up
 * resources such as threads. In the example above, try-with-resources is used, which automatically
 * calls close().
 *
 * <table>
 *    <caption>Methods</caption>
 *    <tr>
 *      <th>Method</th>
 *      <th>Description</th>
 *      <th>Method Variants</th>
 *    </tr>
 *    <tr>
 *      <td><p> CreateSecuritySettings</td>
 *      <td><p> Create security settings in the specified location.</td>
 *      <td>
 *      <p>Request object method variants only take one parameter, a request object, which must be constructed before the call.</p>
 *      <ul>
 *           <li><p> createSecuritySettings(CreateSecuritySettingsRequest request)
 *      </ul>
 *      <p>"Flattened" method variants have converted the fields of the request object into function parameters to enable multiple ways to call the same method.</p>
 *      <ul>
 *           <li><p> createSecuritySettings(LocationName parent, SecuritySettings securitySettings)
 *           <li><p> createSecuritySettings(String parent, SecuritySettings securitySettings)
 *      </ul>
 *      <p>Callable method variants take no parameters and return an immutable API callable object, which can be used to initiate calls to the service.</p>
 *      <ul>
 *           <li><p> createSecuritySettingsCallable()
 *      </ul>
 *       </td>
 *    </tr>
 *    <tr>
 *      <td><p> GetSecuritySettings</td>
 *      <td><p> Retrieves the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings]. The returned settings may be stale by up to 1 minute.</td>
 *      <td>
 *      <p>Request object method variants only take one parameter, a request object, which must be constructed before the call.</p>
 *      <ul>
 *           <li><p> getSecuritySettings(GetSecuritySettingsRequest request)
 *      </ul>
 *      <p>"Flattened" method variants have converted the fields of the request object into function parameters to enable multiple ways to call the same method.</p>
 *      <ul>
 *           <li><p> getSecuritySettings(SecuritySettingsName name)
 *           <li><p> getSecuritySettings(String name)
 *      </ul>
 *      <p>Callable method variants take no parameters and return an immutable API callable object, which can be used to initiate calls to the service.</p>
 *      <ul>
 *           <li><p> getSecuritySettingsCallable()
 *      </ul>
 *       </td>
 *    </tr>
 *    <tr>
 *      <td><p> UpdateSecuritySettings</td>
 *      <td><p> Updates the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings].</td>
 *      <td>
 *      <p>Request object method variants only take one parameter, a request object, which must be constructed before the call.</p>
 *      <ul>
 *           <li><p> updateSecuritySettings(UpdateSecuritySettingsRequest request)
 *      </ul>
 *      <p>"Flattened" method variants have converted the fields of the request object into function parameters to enable multiple ways to call the same method.</p>
 *      <ul>
 *           <li><p> updateSecuritySettings(SecuritySettings securitySettings, FieldMask updateMask)
 *      </ul>
 *      <p>Callable method variants take no parameters and return an immutable API callable object, which can be used to initiate calls to the service.</p>
 *      <ul>
 *           <li><p> updateSecuritySettingsCallable()
 *      </ul>
 *       </td>
 *    </tr>
 *    <tr>
 *      <td><p> ListSecuritySettings</td>
 *      <td><p> Returns the list of all security settings in the specified location.</td>
 *      <td>
 *      <p>Request object method variants only take one parameter, a request object, which must be constructed before the call.</p>
 *      <ul>
 *           <li><p> listSecuritySettings(ListSecuritySettingsRequest request)
 *      </ul>
 *      <p>"Flattened" method variants have converted the fields of the request object into function parameters to enable multiple ways to call the same method.</p>
 *      <ul>
 *           <li><p> listSecuritySettings(LocationName parent)
 *           <li><p> listSecuritySettings(String parent)
 *      </ul>
 *      <p>Callable method variants take no parameters and return an immutable API callable object, which can be used to initiate calls to the service.</p>
 *      <ul>
 *           <li><p> listSecuritySettingsPagedCallable()
 *           <li><p> listSecuritySettingsCallable()
 *      </ul>
 *       </td>
 *    </tr>
 *    <tr>
 *      <td><p> DeleteSecuritySettings</td>
 *      <td><p> Deletes the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings].</td>
 *      <td>
 *      <p>Request object method variants only take one parameter, a request object, which must be constructed before the call.</p>
 *      <ul>
 *           <li><p> deleteSecuritySettings(DeleteSecuritySettingsRequest request)
 *      </ul>
 *      <p>"Flattened" method variants have converted the fields of the request object into function parameters to enable multiple ways to call the same method.</p>
 *      <ul>
 *           <li><p> deleteSecuritySettings(SecuritySettingsName name)
 *           <li><p> deleteSecuritySettings(String name)
 *      </ul>
 *      <p>Callable method variants take no parameters and return an immutable API callable object, which can be used to initiate calls to the service.</p>
 *      <ul>
 *           <li><p> deleteSecuritySettingsCallable()
 *      </ul>
 *       </td>
 *    </tr>
 *    <tr>
 *      <td><p> ListLocations</td>
 *      <td><p> Lists information about the supported locations for this service.</td>
 *      <td>
 *      <p>Request object method variants only take one parameter, a request object, which must be constructed before the call.</p>
 *      <ul>
 *           <li><p> listLocations(ListLocationsRequest request)
 *      </ul>
 *      <p>Callable method variants take no parameters and return an immutable API callable object, which can be used to initiate calls to the service.</p>
 *      <ul>
 *           <li><p> listLocationsPagedCallable()
 *           <li><p> listLocationsCallable()
 *      </ul>
 *       </td>
 *    </tr>
 *    <tr>
 *      <td><p> GetLocation</td>
 *      <td><p> Gets information about a location.</td>
 *      <td>
 *      <p>Request object method variants only take one parameter, a request object, which must be constructed before the call.</p>
 *      <ul>
 *           <li><p> getLocation(GetLocationRequest request)
 *      </ul>
 *      <p>Callable method variants take no parameters and return an immutable API callable object, which can be used to initiate calls to the service.</p>
 *      <ul>
 *           <li><p> getLocationCallable()
 *      </ul>
 *       </td>
 *    </tr>
 *  </table>
 *
 * <p>See the individual methods for example code.
 *
 * <p>Many parameters require resource names to be formatted in a particular way. To assist with
 * these names, this class includes a format method for each type of name, and additionally a parse
 * method to extract the individual identifiers contained within names that are returned.
 *
 * <p>This class can be customized by passing in a custom instance of
 * SecuritySettingsServiceSettings to create(). For example:
 *
 * <p>To customize credentials:
 *
 * <pre>{@code
 * // This snippet has been automatically generated and should be regarded as a code template only.
 * // It will require modifications to work:
 * // - It may require correct/in-range values for request initialization.
 * // - It may require specifying regional endpoints when creating the service client as shown in
 * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
 * SecuritySettingsServiceSettings securitySettingsServiceSettings =
 *     SecuritySettingsServiceSettings.newBuilder()
 *         .setCredentialsProvider(FixedCredentialsProvider.create(myCredentials))
 *         .build();
 * SecuritySettingsServiceClient securitySettingsServiceClient =
 *     SecuritySettingsServiceClient.create(securitySettingsServiceSettings);
 * }</pre>
 *
 * <p>To customize the endpoint:
 *
 * <pre>{@code
 * // This snippet has been automatically generated and should be regarded as a code template only.
 * // It will require modifications to work:
 * // - It may require correct/in-range values for request initialization.
 * // - It may require specifying regional endpoints when creating the service client as shown in
 * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
 * SecuritySettingsServiceSettings securitySettingsServiceSettings =
 *     SecuritySettingsServiceSettings.newBuilder().setEndpoint(myEndpoint).build();
 * SecuritySettingsServiceClient securitySettingsServiceClient =
 *     SecuritySettingsServiceClient.create(securitySettingsServiceSettings);
 * }</pre>
 *
 * <p>To use REST (HTTP1.1/JSON) transport (instead of gRPC) for sending and receiving requests over
 * the wire:
 *
 * <pre>{@code
 * // This snippet has been automatically generated and should be regarded as a code template only.
 * // It will require modifications to work:
 * // - It may require correct/in-range values for request initialization.
 * // - It may require specifying regional endpoints when creating the service client as shown in
 * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
 * SecuritySettingsServiceSettings securitySettingsServiceSettings =
 *     SecuritySettingsServiceSettings.newHttpJsonBuilder().build();
 * SecuritySettingsServiceClient securitySettingsServiceClient =
 *     SecuritySettingsServiceClient.create(securitySettingsServiceSettings);
 * }</pre>
 *
 * <p>Please refer to the GitHub repository's samples for more quickstart code snippets.
 */
@Generated("by gapic-generator-java")
public class SecuritySettingsServiceClient implements BackgroundResource {
  private final SecuritySettingsServiceSettings settings;
  private final SecuritySettingsServiceStub stub;

  /** Constructs an instance of SecuritySettingsServiceClient with default settings. */
  public static final SecuritySettingsServiceClient create() throws IOException {
    return create(SecuritySettingsServiceSettings.newBuilder().build());
  }

  /**
   * Constructs an instance of SecuritySettingsServiceClient, using the given settings. The channels
   * are created based on the settings passed in, or defaults for any settings that are not set.
   */
  public static final SecuritySettingsServiceClient create(SecuritySettingsServiceSettings settings)
      throws IOException {
    return new SecuritySettingsServiceClient(settings);
  }

  /**
   * Constructs an instance of SecuritySettingsServiceClient, using the given stub for making calls.
   * This is for advanced usage - prefer using create(SecuritySettingsServiceSettings).
   */
  public static final SecuritySettingsServiceClient create(SecuritySettingsServiceStub stub) {
    return new SecuritySettingsServiceClient(stub);
  }

  /**
   * Constructs an instance of SecuritySettingsServiceClient, using the given settings. This is
   * protected so that it is easy to make a subclass, but otherwise, the static factory methods
   * should be preferred.
   */
  protected SecuritySettingsServiceClient(SecuritySettingsServiceSettings settings)
      throws IOException {
    this.settings = settings;
    this.stub = ((SecuritySettingsServiceStubSettings) settings.getStubSettings()).createStub();
  }

  protected SecuritySettingsServiceClient(SecuritySettingsServiceStub stub) {
    this.settings = null;
    this.stub = stub;
  }

  public final SecuritySettingsServiceSettings getSettings() {
    return settings;
  }

  public SecuritySettingsServiceStub getStub() {
    return stub;
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Create security settings in the specified location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   LocationName parent = LocationName.of("[PROJECT]", "[LOCATION]");
   *   SecuritySettings securitySettings = SecuritySettings.newBuilder().build();
   *   SecuritySettings response =
   *       securitySettingsServiceClient.createSecuritySettings(parent, securitySettings);
   * }
   * }</pre>
   *
   * @param parent Required. The location to create an
   *     [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings] for. Format:
   *     `projects/&lt;ProjectID&gt;/locations/&lt;LocationID&gt;`.
   * @param securitySettings Required. The security settings to create.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final SecuritySettings createSecuritySettings(
      LocationName parent, SecuritySettings securitySettings) {
    CreateSecuritySettingsRequest request =
        CreateSecuritySettingsRequest.newBuilder()
            .setParent(parent == null ? null : parent.toString())
            .setSecuritySettings(securitySettings)
            .build();
    return createSecuritySettings(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Create security settings in the specified location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   String parent = LocationName.of("[PROJECT]", "[LOCATION]").toString();
   *   SecuritySettings securitySettings = SecuritySettings.newBuilder().build();
   *   SecuritySettings response =
   *       securitySettingsServiceClient.createSecuritySettings(parent, securitySettings);
   * }
   * }</pre>
   *
   * @param parent Required. The location to create an
   *     [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings] for. Format:
   *     `projects/&lt;ProjectID&gt;/locations/&lt;LocationID&gt;`.
   * @param securitySettings Required. The security settings to create.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final SecuritySettings createSecuritySettings(
      String parent, SecuritySettings securitySettings) {
    CreateSecuritySettingsRequest request =
        CreateSecuritySettingsRequest.newBuilder()
            .setParent(parent)
            .setSecuritySettings(securitySettings)
            .build();
    return createSecuritySettings(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Create security settings in the specified location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   CreateSecuritySettingsRequest request =
   *       CreateSecuritySettingsRequest.newBuilder()
   *           .setParent(LocationName.of("[PROJECT]", "[LOCATION]").toString())
   *           .setSecuritySettings(SecuritySettings.newBuilder().build())
   *           .build();
   *   SecuritySettings response = securitySettingsServiceClient.createSecuritySettings(request);
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final SecuritySettings createSecuritySettings(CreateSecuritySettingsRequest request) {
    return createSecuritySettingsCallable().call(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Create security settings in the specified location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   CreateSecuritySettingsRequest request =
   *       CreateSecuritySettingsRequest.newBuilder()
   *           .setParent(LocationName.of("[PROJECT]", "[LOCATION]").toString())
   *           .setSecuritySettings(SecuritySettings.newBuilder().build())
   *           .build();
   *   ApiFuture<SecuritySettings> future =
   *       securitySettingsServiceClient.createSecuritySettingsCallable().futureCall(request);
   *   // Do something.
   *   SecuritySettings response = future.get();
   * }
   * }</pre>
   */
  public final UnaryCallable<CreateSecuritySettingsRequest, SecuritySettings>
      createSecuritySettingsCallable() {
    return stub.createSecuritySettingsCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings]. The
   * returned settings may be stale by up to 1 minute.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   SecuritySettingsName name =
   *       SecuritySettingsName.of("[PROJECT]", "[LOCATION]", "[SECURITY_SETTINGS]");
   *   SecuritySettings response = securitySettingsServiceClient.getSecuritySettings(name);
   * }
   * }</pre>
   *
   * @param name Required. Resource name of the settings. Format:
   *     `projects/&lt;ProjectID&gt;/locations/&lt;LocationID&gt;/securitySettings/&lt;securitysettingsID&gt;`.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final SecuritySettings getSecuritySettings(SecuritySettingsName name) {
    GetSecuritySettingsRequest request =
        GetSecuritySettingsRequest.newBuilder()
            .setName(name == null ? null : name.toString())
            .build();
    return getSecuritySettings(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings]. The
   * returned settings may be stale by up to 1 minute.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   String name =
   *       SecuritySettingsName.of("[PROJECT]", "[LOCATION]", "[SECURITY_SETTINGS]").toString();
   *   SecuritySettings response = securitySettingsServiceClient.getSecuritySettings(name);
   * }
   * }</pre>
   *
   * @param name Required. Resource name of the settings. Format:
   *     `projects/&lt;ProjectID&gt;/locations/&lt;LocationID&gt;/securitySettings/&lt;securitysettingsID&gt;`.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final SecuritySettings getSecuritySettings(String name) {
    GetSecuritySettingsRequest request =
        GetSecuritySettingsRequest.newBuilder().setName(name).build();
    return getSecuritySettings(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings]. The
   * returned settings may be stale by up to 1 minute.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   GetSecuritySettingsRequest request =
   *       GetSecuritySettingsRequest.newBuilder()
   *           .setName(
   *               SecuritySettingsName.of("[PROJECT]", "[LOCATION]", "[SECURITY_SETTINGS]")
   *                   .toString())
   *           .build();
   *   SecuritySettings response = securitySettingsServiceClient.getSecuritySettings(request);
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final SecuritySettings getSecuritySettings(GetSecuritySettingsRequest request) {
    return getSecuritySettingsCallable().call(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Retrieves the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings]. The
   * returned settings may be stale by up to 1 minute.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   GetSecuritySettingsRequest request =
   *       GetSecuritySettingsRequest.newBuilder()
   *           .setName(
   *               SecuritySettingsName.of("[PROJECT]", "[LOCATION]", "[SECURITY_SETTINGS]")
   *                   .toString())
   *           .build();
   *   ApiFuture<SecuritySettings> future =
   *       securitySettingsServiceClient.getSecuritySettingsCallable().futureCall(request);
   *   // Do something.
   *   SecuritySettings response = future.get();
   * }
   * }</pre>
   */
  public final UnaryCallable<GetSecuritySettingsRequest, SecuritySettings>
      getSecuritySettingsCallable() {
    return stub.getSecuritySettingsCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings].
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   SecuritySettings securitySettings = SecuritySettings.newBuilder().build();
   *   FieldMask updateMask = FieldMask.newBuilder().build();
   *   SecuritySettings response =
   *       securitySettingsServiceClient.updateSecuritySettings(securitySettings, updateMask);
   * }
   * }</pre>
   *
   * @param securitySettings Required. [SecuritySettings] object that contains values for each of
   *     the fields to update.
   * @param updateMask Required. The mask to control which fields get updated. If the mask is not
   *     present, all fields will be updated.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final SecuritySettings updateSecuritySettings(
      SecuritySettings securitySettings, FieldMask updateMask) {
    UpdateSecuritySettingsRequest request =
        UpdateSecuritySettingsRequest.newBuilder()
            .setSecuritySettings(securitySettings)
            .setUpdateMask(updateMask)
            .build();
    return updateSecuritySettings(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings].
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   UpdateSecuritySettingsRequest request =
   *       UpdateSecuritySettingsRequest.newBuilder()
   *           .setSecuritySettings(SecuritySettings.newBuilder().build())
   *           .setUpdateMask(FieldMask.newBuilder().build())
   *           .build();
   *   SecuritySettings response = securitySettingsServiceClient.updateSecuritySettings(request);
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final SecuritySettings updateSecuritySettings(UpdateSecuritySettingsRequest request) {
    return updateSecuritySettingsCallable().call(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Updates the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings].
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   UpdateSecuritySettingsRequest request =
   *       UpdateSecuritySettingsRequest.newBuilder()
   *           .setSecuritySettings(SecuritySettings.newBuilder().build())
   *           .setUpdateMask(FieldMask.newBuilder().build())
   *           .build();
   *   ApiFuture<SecuritySettings> future =
   *       securitySettingsServiceClient.updateSecuritySettingsCallable().futureCall(request);
   *   // Do something.
   *   SecuritySettings response = future.get();
   * }
   * }</pre>
   */
  public final UnaryCallable<UpdateSecuritySettingsRequest, SecuritySettings>
      updateSecuritySettingsCallable() {
    return stub.updateSecuritySettingsCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Returns the list of all security settings in the specified location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   LocationName parent = LocationName.of("[PROJECT]", "[LOCATION]");
   *   for (SecuritySettings element :
   *       securitySettingsServiceClient.listSecuritySettings(parent).iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   *
   * @param parent Required. The location to list all security settings for. Format:
   *     `projects/&lt;ProjectID&gt;/locations/&lt;LocationID&gt;`.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final ListSecuritySettingsPagedResponse listSecuritySettings(LocationName parent) {
    ListSecuritySettingsRequest request =
        ListSecuritySettingsRequest.newBuilder()
            .setParent(parent == null ? null : parent.toString())
            .build();
    return listSecuritySettings(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Returns the list of all security settings in the specified location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   String parent = LocationName.of("[PROJECT]", "[LOCATION]").toString();
   *   for (SecuritySettings element :
   *       securitySettingsServiceClient.listSecuritySettings(parent).iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   *
   * @param parent Required. The location to list all security settings for. Format:
   *     `projects/&lt;ProjectID&gt;/locations/&lt;LocationID&gt;`.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final ListSecuritySettingsPagedResponse listSecuritySettings(String parent) {
    ListSecuritySettingsRequest request =
        ListSecuritySettingsRequest.newBuilder().setParent(parent).build();
    return listSecuritySettings(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Returns the list of all security settings in the specified location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   ListSecuritySettingsRequest request =
   *       ListSecuritySettingsRequest.newBuilder()
   *           .setParent(LocationName.of("[PROJECT]", "[LOCATION]").toString())
   *           .setPageSize(883849137)
   *           .setPageToken("pageToken873572522")
   *           .build();
   *   for (SecuritySettings element :
   *       securitySettingsServiceClient.listSecuritySettings(request).iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final ListSecuritySettingsPagedResponse listSecuritySettings(
      ListSecuritySettingsRequest request) {
    return listSecuritySettingsPagedCallable().call(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Returns the list of all security settings in the specified location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   ListSecuritySettingsRequest request =
   *       ListSecuritySettingsRequest.newBuilder()
   *           .setParent(LocationName.of("[PROJECT]", "[LOCATION]").toString())
   *           .setPageSize(883849137)
   *           .setPageToken("pageToken873572522")
   *           .build();
   *   ApiFuture<SecuritySettings> future =
   *       securitySettingsServiceClient.listSecuritySettingsPagedCallable().futureCall(request);
   *   // Do something.
   *   for (SecuritySettings element : future.get().iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   */
  public final UnaryCallable<ListSecuritySettingsRequest, ListSecuritySettingsPagedResponse>
      listSecuritySettingsPagedCallable() {
    return stub.listSecuritySettingsPagedCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Returns the list of all security settings in the specified location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   ListSecuritySettingsRequest request =
   *       ListSecuritySettingsRequest.newBuilder()
   *           .setParent(LocationName.of("[PROJECT]", "[LOCATION]").toString())
   *           .setPageSize(883849137)
   *           .setPageToken("pageToken873572522")
   *           .build();
   *   while (true) {
   *     ListSecuritySettingsResponse response =
   *         securitySettingsServiceClient.listSecuritySettingsCallable().call(request);
   *     for (SecuritySettings element : response.getSecuritySettingsList()) {
   *       // doThingsWith(element);
   *     }
   *     String nextPageToken = response.getNextPageToken();
   *     if (!Strings.isNullOrEmpty(nextPageToken)) {
   *       request = request.toBuilder().setPageToken(nextPageToken).build();
   *     } else {
   *       break;
   *     }
   *   }
   * }
   * }</pre>
   */
  public final UnaryCallable<ListSecuritySettingsRequest, ListSecuritySettingsResponse>
      listSecuritySettingsCallable() {
    return stub.listSecuritySettingsCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Deletes the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings].
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   SecuritySettingsName name =
   *       SecuritySettingsName.of("[PROJECT]", "[LOCATION]", "[SECURITY_SETTINGS]");
   *   securitySettingsServiceClient.deleteSecuritySettings(name);
   * }
   * }</pre>
   *
   * @param name Required. The name of the
   *     [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings] to delete. Format:
   *     `projects/&lt;ProjectID&gt;/locations/&lt;LocationID&gt;/securitySettings/&lt;SecuritySettingsID&gt;`.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final void deleteSecuritySettings(SecuritySettingsName name) {
    DeleteSecuritySettingsRequest request =
        DeleteSecuritySettingsRequest.newBuilder()
            .setName(name == null ? null : name.toString())
            .build();
    deleteSecuritySettings(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Deletes the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings].
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   String name =
   *       SecuritySettingsName.of("[PROJECT]", "[LOCATION]", "[SECURITY_SETTINGS]").toString();
   *   securitySettingsServiceClient.deleteSecuritySettings(name);
   * }
   * }</pre>
   *
   * @param name Required. The name of the
   *     [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings] to delete. Format:
   *     `projects/&lt;ProjectID&gt;/locations/&lt;LocationID&gt;/securitySettings/&lt;SecuritySettingsID&gt;`.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final void deleteSecuritySettings(String name) {
    DeleteSecuritySettingsRequest request =
        DeleteSecuritySettingsRequest.newBuilder().setName(name).build();
    deleteSecuritySettings(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Deletes the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings].
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   DeleteSecuritySettingsRequest request =
   *       DeleteSecuritySettingsRequest.newBuilder()
   *           .setName(
   *               SecuritySettingsName.of("[PROJECT]", "[LOCATION]", "[SECURITY_SETTINGS]")
   *                   .toString())
   *           .build();
   *   securitySettingsServiceClient.deleteSecuritySettings(request);
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final void deleteSecuritySettings(DeleteSecuritySettingsRequest request) {
    deleteSecuritySettingsCallable().call(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Deletes the specified [SecuritySettings][google.cloud.dialogflow.cx.v3.SecuritySettings].
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   DeleteSecuritySettingsRequest request =
   *       DeleteSecuritySettingsRequest.newBuilder()
   *           .setName(
   *               SecuritySettingsName.of("[PROJECT]", "[LOCATION]", "[SECURITY_SETTINGS]")
   *                   .toString())
   *           .build();
   *   ApiFuture<Empty> future =
   *       securitySettingsServiceClient.deleteSecuritySettingsCallable().futureCall(request);
   *   // Do something.
   *   future.get();
   * }
   * }</pre>
   */
  public final UnaryCallable<DeleteSecuritySettingsRequest, Empty>
      deleteSecuritySettingsCallable() {
    return stub.deleteSecuritySettingsCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Lists information about the supported locations for this service.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   ListLocationsRequest request =
   *       ListLocationsRequest.newBuilder()
   *           .setName("name3373707")
   *           .setFilter("filter-1274492040")
   *           .setPageSize(883849137)
   *           .setPageToken("pageToken873572522")
   *           .build();
   *   for (Location element : securitySettingsServiceClient.listLocations(request).iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final ListLocationsPagedResponse listLocations(ListLocationsRequest request) {
    return listLocationsPagedCallable().call(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Lists information about the supported locations for this service.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   ListLocationsRequest request =
   *       ListLocationsRequest.newBuilder()
   *           .setName("name3373707")
   *           .setFilter("filter-1274492040")
   *           .setPageSize(883849137)
   *           .setPageToken("pageToken873572522")
   *           .build();
   *   ApiFuture<Location> future =
   *       securitySettingsServiceClient.listLocationsPagedCallable().futureCall(request);
   *   // Do something.
   *   for (Location element : future.get().iterateAll()) {
   *     // doThingsWith(element);
   *   }
   * }
   * }</pre>
   */
  public final UnaryCallable<ListLocationsRequest, ListLocationsPagedResponse>
      listLocationsPagedCallable() {
    return stub.listLocationsPagedCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Lists information about the supported locations for this service.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   ListLocationsRequest request =
   *       ListLocationsRequest.newBuilder()
   *           .setName("name3373707")
   *           .setFilter("filter-1274492040")
   *           .setPageSize(883849137)
   *           .setPageToken("pageToken873572522")
   *           .build();
   *   while (true) {
   *     ListLocationsResponse response =
   *         securitySettingsServiceClient.listLocationsCallable().call(request);
   *     for (Location element : response.getLocationsList()) {
   *       // doThingsWith(element);
   *     }
   *     String nextPageToken = response.getNextPageToken();
   *     if (!Strings.isNullOrEmpty(nextPageToken)) {
   *       request = request.toBuilder().setPageToken(nextPageToken).build();
   *     } else {
   *       break;
   *     }
   *   }
   * }
   * }</pre>
   */
  public final UnaryCallable<ListLocationsRequest, ListLocationsResponse> listLocationsCallable() {
    return stub.listLocationsCallable();
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Gets information about a location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   GetLocationRequest request = GetLocationRequest.newBuilder().setName("name3373707").build();
   *   Location response = securitySettingsServiceClient.getLocation(request);
   * }
   * }</pre>
   *
   * @param request The request object containing all of the parameters for the API call.
   * @throws com.google.api.gax.rpc.ApiException if the remote call fails
   */
  public final Location getLocation(GetLocationRequest request) {
    return getLocationCallable().call(request);
  }

  // AUTO-GENERATED DOCUMENTATION AND METHOD.
  /**
   * Gets information about a location.
   *
   * <p>Sample code:
   *
   * <pre>{@code
   * // This snippet has been automatically generated and should be regarded as a code template only.
   * // It will require modifications to work:
   * // - It may require correct/in-range values for request initialization.
   * // - It may require specifying regional endpoints when creating the service client as shown in
   * // https://cloud.google.com/java/docs/setup#configure_endpoints_for_the_client_library
   * try (SecuritySettingsServiceClient securitySettingsServiceClient =
   *     SecuritySettingsServiceClient.create()) {
   *   GetLocationRequest request = GetLocationRequest.newBuilder().setName("name3373707").build();
   *   ApiFuture<Location> future =
   *       securitySettingsServiceClient.getLocationCallable().futureCall(request);
   *   // Do something.
   *   Location response = future.get();
   * }
   * }</pre>
   */
  public final UnaryCallable<GetLocationRequest, Location> getLocationCallable() {
    return stub.getLocationCallable();
  }

  @Override
  public final void close() {
    stub.close();
  }

  @Override
  public void shutdown() {
    stub.shutdown();
  }

  @Override
  public boolean isShutdown() {
    return stub.isShutdown();
  }

  @Override
  public boolean isTerminated() {
    return stub.isTerminated();
  }

  @Override
  public void shutdownNow() {
    stub.shutdownNow();
  }

  @Override
  public boolean awaitTermination(long duration, TimeUnit unit) throws InterruptedException {
    return stub.awaitTermination(duration, unit);
  }

  public static class ListSecuritySettingsPagedResponse
      extends AbstractPagedListResponse<
          ListSecuritySettingsRequest,
          ListSecuritySettingsResponse,
          SecuritySettings,
          ListSecuritySettingsPage,
          ListSecuritySettingsFixedSizeCollection> {

    public static ApiFuture<ListSecuritySettingsPagedResponse> createAsync(
        PageContext<ListSecuritySettingsRequest, ListSecuritySettingsResponse, SecuritySettings>
            context,
        ApiFuture<ListSecuritySettingsResponse> futureResponse) {
      ApiFuture<ListSecuritySettingsPage> futurePage =
          ListSecuritySettingsPage.createEmptyPage().createPageAsync(context, futureResponse);
      return ApiFutures.transform(
          futurePage,
          input -> new ListSecuritySettingsPagedResponse(input),
          MoreExecutors.directExecutor());
    }

    private ListSecuritySettingsPagedResponse(ListSecuritySettingsPage page) {
      super(page, ListSecuritySettingsFixedSizeCollection.createEmptyCollection());
    }
  }

  public static class ListSecuritySettingsPage
      extends AbstractPage<
          ListSecuritySettingsRequest,
          ListSecuritySettingsResponse,
          SecuritySettings,
          ListSecuritySettingsPage> {

    private ListSecuritySettingsPage(
        PageContext<ListSecuritySettingsRequest, ListSecuritySettingsResponse, SecuritySettings>
            context,
        ListSecuritySettingsResponse response) {
      super(context, response);
    }

    private static ListSecuritySettingsPage createEmptyPage() {
      return new ListSecuritySettingsPage(null, null);
    }

    @Override
    protected ListSecuritySettingsPage createPage(
        PageContext<ListSecuritySettingsRequest, ListSecuritySettingsResponse, SecuritySettings>
            context,
        ListSecuritySettingsResponse response) {
      return new ListSecuritySettingsPage(context, response);
    }

    @Override
    public ApiFuture<ListSecuritySettingsPage> createPageAsync(
        PageContext<ListSecuritySettingsRequest, ListSecuritySettingsResponse, SecuritySettings>
            context,
        ApiFuture<ListSecuritySettingsResponse> futureResponse) {
      return super.createPageAsync(context, futureResponse);
    }
  }

  public static class ListSecuritySettingsFixedSizeCollection
      extends AbstractFixedSizeCollection<
          ListSecuritySettingsRequest,
          ListSecuritySettingsResponse,
          SecuritySettings,
          ListSecuritySettingsPage,
          ListSecuritySettingsFixedSizeCollection> {

    private ListSecuritySettingsFixedSizeCollection(
        List<ListSecuritySettingsPage> pages, int collectionSize) {
      super(pages, collectionSize);
    }

    private static ListSecuritySettingsFixedSizeCollection createEmptyCollection() {
      return new ListSecuritySettingsFixedSizeCollection(null, 0);
    }

    @Override
    protected ListSecuritySettingsFixedSizeCollection createCollection(
        List<ListSecuritySettingsPage> pages, int collectionSize) {
      return new ListSecuritySettingsFixedSizeCollection(pages, collectionSize);
    }
  }

  public static class ListLocationsPagedResponse
      extends AbstractPagedListResponse<
          ListLocationsRequest,
          ListLocationsResponse,
          Location,
          ListLocationsPage,
          ListLocationsFixedSizeCollection> {

    public static ApiFuture<ListLocationsPagedResponse> createAsync(
        PageContext<ListLocationsRequest, ListLocationsResponse, Location> context,
        ApiFuture<ListLocationsResponse> futureResponse) {
      ApiFuture<ListLocationsPage> futurePage =
          ListLocationsPage.createEmptyPage().createPageAsync(context, futureResponse);
      return ApiFutures.transform(
          futurePage,
          input -> new ListLocationsPagedResponse(input),
          MoreExecutors.directExecutor());
    }

    private ListLocationsPagedResponse(ListLocationsPage page) {
      super(page, ListLocationsFixedSizeCollection.createEmptyCollection());
    }
  }

  public static class ListLocationsPage
      extends AbstractPage<
          ListLocationsRequest, ListLocationsResponse, Location, ListLocationsPage> {

    private ListLocationsPage(
        PageContext<ListLocationsRequest, ListLocationsResponse, Location> context,
        ListLocationsResponse response) {
      super(context, response);
    }

    private static ListLocationsPage createEmptyPage() {
      return new ListLocationsPage(null, null);
    }

    @Override
    protected ListLocationsPage createPage(
        PageContext<ListLocationsRequest, ListLocationsResponse, Location> context,
        ListLocationsResponse response) {
      return new ListLocationsPage(context, response);
    }

    @Override
    public ApiFuture<ListLocationsPage> createPageAsync(
        PageContext<ListLocationsRequest, ListLocationsResponse, Location> context,
        ApiFuture<ListLocationsResponse> futureResponse) {
      return super.createPageAsync(context, futureResponse);
    }
  }

  public static class ListLocationsFixedSizeCollection
      extends AbstractFixedSizeCollection<
          ListLocationsRequest,
          ListLocationsResponse,
          Location,
          ListLocationsPage,
          ListLocationsFixedSizeCollection> {

    private ListLocationsFixedSizeCollection(List<ListLocationsPage> pages, int collectionSize) {
      super(pages, collectionSize);
    }

    private static ListLocationsFixedSizeCollection createEmptyCollection() {
      return new ListLocationsFixedSizeCollection(null, 0);
    }

    @Override
    protected ListLocationsFixedSizeCollection createCollection(
        List<ListLocationsPage> pages, int collectionSize) {
      return new ListLocationsFixedSizeCollection(pages, collectionSize);
    }
  }
}
