/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio;

import com.github.rholder.retry.Retryer;
import com.github.rholder.retry.RetryerBuilder;
import com.github.rholder.retry.StopStrategies;
import com.github.rholder.retry.WaitStrategies;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageExceptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.StringPaths;
import com.google.common.base.Preconditions;
import com.google.common.flogger.GoogleLogger;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import com.google.google.storage.v1.GetObjectMediaRequest;
import com.google.google.storage.v1.GetObjectMediaResponse;
import com.google.google.storage.v1.GetObjectRequest;
import com.google.google.storage.v1.StorageGrpc;
import com.google.protobuf.ByteString;
import io.grpc.Context;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import java.io.EOFException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SeekableByteChannel;
import java.time.Duration;
import java.util.Iterator;
import java.util.OptionalLong;
import java.util.concurrent.TimeUnit;
import javax.annotation.Nullable;

public class GoogleCloudStorageGrpcReadChannel
implements SeekableByteChannel {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    private static final Duration READ_STREAM_TIMEOUT = Duration.ofMinutes(20L);
    private static final Duration READ_OBJECT_METADATA_TIMEOUT = Duration.ofMinutes(1L);
    private static final int READ_RETRIES = 5;
    @Nullable
    Context.CancellableContext requestContext;
    GoogleCloudStorageReadOptions.Fadvise readStrategy;
    private StorageGrpc.StorageBlockingStub stub;
    private String bucketName;
    private String objectName;
    private long objectGeneration;
    private long objectSize;
    private boolean channelIsOpen = true;
    private long position = 0L;
    private long bytesToSkipBeforeReading = 0L;
    @Nullable
    private ByteString bufferedContent = null;
    private int bufferedContentReadOffset = 0;
    @Nullable
    private Iterator<GetObjectMediaResponse> resIterator = null;
    private GoogleCloudStorageReadOptions readOptions;

    private GoogleCloudStorageGrpcReadChannel(StorageGrpc.StorageBlockingStub gcsGrpcBlockingStub, String bucketName, String objectName, long objectGeneration, long objectSize, GoogleCloudStorageReadOptions readOptions) {
        this.stub = gcsGrpcBlockingStub;
        this.bucketName = bucketName;
        this.objectName = objectName;
        this.objectGeneration = objectGeneration;
        this.objectSize = objectSize;
        this.readOptions = readOptions;
        this.readStrategy = readOptions.getFadvise();
    }

    public static GoogleCloudStorageGrpcReadChannel open(StorageGrpc.StorageBlockingStub stub, String bucketName, String objectName, GoogleCloudStorageReadOptions readOptions) throws IOException {
        com.google.google.storage.v1.Object getObjectResult;
        try {
            getObjectResult = ((StorageGrpc.StorageBlockingStub)stub.withDeadlineAfter(READ_OBJECT_METADATA_TIMEOUT.toMillis(), TimeUnit.MILLISECONDS)).getObject(GetObjectRequest.newBuilder().setBucket(bucketName).setObject(objectName).build());
        }
        catch (StatusRuntimeException e) {
            throw GoogleCloudStorageGrpcReadChannel.convertError(e, bucketName, objectName);
        }
        if (getObjectResult.getContentEncoding().contains("gzip")) {
            throw new IOException("Can't read GZIP encoded files - content encoding support is disabled.");
        }
        return new GoogleCloudStorageGrpcReadChannel(stub, bucketName, objectName, getObjectResult.getGeneration(), getObjectResult.getSize(), readOptions);
    }

    private static IOException convertError(StatusRuntimeException error, String bucketName, String objectName) {
        Status.Code statusCode = Status.fromThrowable((Throwable)error).getCode();
        String msg = String.format("Error reading '%s'", StringPaths.fromComponents(bucketName, objectName));
        if (statusCode == Status.Code.NOT_FOUND) {
            return GoogleCloudStorageExceptions.createFileNotFoundException(bucketName, objectName, new IOException(msg, error));
        }
        if (statusCode == Status.Code.OUT_OF_RANGE) {
            return (IOException)new EOFException(msg).initCause(error);
        }
        return new IOException(msg, error);
    }

    private static final void put(ByteString source, int offset, int size, ByteBuffer dest) {
        ByteString croppedSource = source.substring(offset, offset + size);
        for (ByteBuffer sourcePiece : croppedSource.asReadOnlyByteBufferList()) {
            dest.put(sourcePiece);
        }
    }

    private int readBufferedContentInto(ByteBuffer byteBuffer) {
        long bufferSkip = Math.min((long)(this.bufferedContent.size() - this.bufferedContentReadOffset), this.bytesToSkipBeforeReading);
        bufferSkip = Math.max(0L, bufferSkip);
        this.bufferedContentReadOffset = (int)((long)this.bufferedContentReadOffset + bufferSkip);
        this.bytesToSkipBeforeReading -= bufferSkip;
        int remainingBufferedBytes = this.bufferedContent.size() - this.bufferedContentReadOffset;
        boolean remainingBufferedContentLargerThanByteBuffer = remainingBufferedBytes > byteBuffer.remaining();
        int bytesToWrite = remainingBufferedContentLargerThanByteBuffer ? byteBuffer.remaining() : remainingBufferedBytes;
        GoogleCloudStorageGrpcReadChannel.put(this.bufferedContent, this.bufferedContentReadOffset, bytesToWrite, byteBuffer);
        this.position += (long)bytesToWrite;
        if (remainingBufferedContentLargerThanByteBuffer) {
            this.bufferedContentReadOffset += bytesToWrite;
        } else {
            this.bufferedContent = null;
            this.bufferedContentReadOffset = 0;
        }
        return bytesToWrite;
    }

    @Override
    public int read(ByteBuffer byteBuffer) throws IOException {
        ((GoogleLogger.Api)logger.atFine()).log("GCS gRPC read request for up to %d bytes at offset %d from object %s", (Object)byteBuffer.remaining(), (Object)this.position(), (Object)this.objectName);
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
        int bytesRead = 0;
        if (this.bufferedContent != null) {
            bytesRead += this.readBufferedContentInto(byteBuffer);
        }
        if (!byteBuffer.hasRemaining()) {
            return bytesRead;
        }
        if (this.position == this.objectSize) {
            return bytesRead > 0 ? bytesRead : -1;
        }
        if (this.resIterator == null) {
            OptionalLong bytesToRead;
            if (this.readStrategy == GoogleCloudStorageReadOptions.Fadvise.RANDOM) {
                long randomRangeSize = Math.max(byteBuffer.remaining(), this.readOptions.getMinRangeRequestSize());
                bytesToRead = OptionalLong.of(randomRangeSize);
            } else {
                bytesToRead = OptionalLong.empty();
            }
            this.requestObjectMedia(bytesToRead);
        }
        while (this.moreServerContent() && byteBuffer.hasRemaining()) {
            GetObjectMediaResponse res = this.resIterator.next();
            ByteString content = res.getChecksummedData().getContent();
            if (this.bytesToSkipBeforeReading >= 0L && this.bytesToSkipBeforeReading < (long)content.size()) {
                content = res.getChecksummedData().getContent().substring((int)this.bytesToSkipBeforeReading);
                this.bytesToSkipBeforeReading = 0L;
            } else if (this.bytesToSkipBeforeReading >= (long)content.size()) {
                this.bytesToSkipBeforeReading -= (long)content.size();
                continue;
            }
            if (this.readOptions.getGrpcChecksumsEnabled() && res.getChecksummedData().hasCrc32C()) {
                Hasher messageHasher = Hashing.crc32c().newHasher();
                messageHasher.putBytes(res.getChecksummedData().getContent().toByteArray());
                int calculatedChecksum = messageHasher.hash().asInt();
                int expectedChecksum = res.getChecksummedData().getCrc32C().getValue();
                if (calculatedChecksum != expectedChecksum) {
                    throw new IOException(String.format("For %s: Message checksum didn't match. Expected %s, got %s.", this, expectedChecksum, calculatedChecksum));
                }
            }
            boolean responseSizeLargerThanRemainingBuffer = content.size() > byteBuffer.remaining();
            int bytesToWrite = responseSizeLargerThanRemainingBuffer ? byteBuffer.remaining() : content.size();
            GoogleCloudStorageGrpcReadChannel.put(content, 0, bytesToWrite, byteBuffer);
            bytesRead += bytesToWrite;
            this.position += (long)bytesToWrite;
            if (!responseSizeLargerThanRemainingBuffer) continue;
            this.bufferedContent = content;
            this.bufferedContentReadOffset = bytesToWrite;
        }
        return bytesRead;
    }

    private void requestObjectMedia(OptionalLong bytesToRead) throws IOException {
        GetObjectMediaRequest.Builder requestBuilder = GetObjectMediaRequest.newBuilder().setBucket(this.bucketName).setObject(this.objectName).setGeneration(this.objectGeneration).setReadOffset(this.position);
        if (bytesToRead.isPresent()) {
            requestBuilder.setReadLimit(bytesToRead.getAsLong());
        }
        GetObjectMediaRequest request = requestBuilder.build();
        Retryer<Boolean> retryer = this.getRetryer();
        try {
            retryer.call(() -> {
                try {
                    this.requestContext = Context.current().withCancellation();
                    Context toReattach = this.requestContext.attach();
                    try {
                        this.resIterator = ((StorageGrpc.StorageBlockingStub)this.stub.withDeadlineAfter(READ_STREAM_TIMEOUT.toMillis(), TimeUnit.MILLISECONDS)).getObjectMedia(request);
                    }
                    finally {
                        this.requestContext.detach(toReattach);
                    }
                }
                catch (StatusRuntimeException e) {
                    throw GoogleCloudStorageGrpcReadChannel.convertError(e, this.bucketName, this.objectName);
                }
                return null;
            });
        }
        catch (Exception e) {
            throw new IOException(String.format("Error reading '%s'", StringPaths.fromComponents(this.bucketName, this.objectName)), e);
        }
    }

    private void cancelCurrentRequest() {
        if (this.requestContext != null) {
            this.requestContext.close();
            this.requestContext = null;
        }
        if (this.resIterator != null) {
            this.resIterator = null;
        }
    }

    private boolean moreServerContent() throws IOException {
        if (this.resIterator == null || this.requestContext == null || this.requestContext.isCancelled()) {
            return false;
        }
        Retryer<Boolean> retryer = this.getRetryer();
        try {
            return (Boolean)retryer.call(() -> {
                try {
                    boolean moreDataAvailable = this.resIterator.hasNext();
                    if (!moreDataAvailable) {
                        this.cancelCurrentRequest();
                    }
                    return moreDataAvailable;
                }
                catch (StatusRuntimeException e) {
                    throw GoogleCloudStorageGrpcReadChannel.convertError(e, this.bucketName, this.objectName);
                }
            });
        }
        catch (Exception e) {
            this.cancelCurrentRequest();
            throw new IOException(String.format("Error reading '%s'", StringPaths.fromComponents(this.bucketName, this.objectName)), e);
        }
    }

    private Retryer<Boolean> getRetryer() {
        return RetryerBuilder.newBuilder().retryIfExceptionOfType(IOException.class).withWaitStrategy(WaitStrategies.exponentialWait((long)2L, (long)20L, (TimeUnit)TimeUnit.SECONDS)).withStopStrategy(StopStrategies.stopAfterAttempt((int)5)).build();
    }

    @Override
    public int write(ByteBuffer byteBuffer) {
        throw new UnsupportedOperationException("Cannot mutate read-only channel: " + this);
    }

    @Override
    public long position() throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
        return this.position + this.bytesToSkipBeforeReading;
    }

    private String resourceIdString() {
        return StringPaths.fromComponents(this.bucketName, this.objectName);
    }

    @Override
    public SeekableByteChannel position(long newPosition) throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
        Preconditions.checkArgument((newPosition >= 0L ? 1 : 0) != 0, (String)"Read position must be non-negative, but was %s", (long)newPosition);
        Preconditions.checkArgument((newPosition < this.size() ? 1 : 0) != 0, (String)"Read position must be before end of file (%s), but was %s", (long)this.size(), (long)newPosition);
        if (newPosition == this.position) {
            return this;
        }
        long seekDistance = newPosition - this.position;
        if (seekDistance >= 0L && seekDistance <= this.readOptions.getInplaceSeekLimit()) {
            this.bytesToSkipBeforeReading = seekDistance;
            return this;
        }
        if (this.readStrategy == GoogleCloudStorageReadOptions.Fadvise.AUTO && (seekDistance < 0L || seekDistance > this.readOptions.getInplaceSeekLimit())) {
            this.readStrategy = GoogleCloudStorageReadOptions.Fadvise.RANDOM;
        }
        this.cancelCurrentRequest();
        this.bufferedContent = null;
        this.bufferedContentReadOffset = 0;
        this.bytesToSkipBeforeReading = 0L;
        this.position = newPosition;
        return this;
    }

    @Override
    public long size() throws IOException {
        if (!this.isOpen()) {
            throw new ClosedChannelException();
        }
        return this.objectSize;
    }

    @Override
    public SeekableByteChannel truncate(long l) throws IOException {
        throw new UnsupportedOperationException("Cannot mutate read-only channel");
    }

    @Override
    public boolean isOpen() {
        return this.channelIsOpen;
    }

    @Override
    public void close() {
        this.cancelCurrentRequest();
        this.channelIsOpen = false;
    }

    public String toString() {
        return "GoogleCloudStorageGrpcReadChannel for bucket: " + this.bucketName + ", object: " + this.objectName + ", generation: " + this.objectGeneration;
    }
}

