/**
 * 
 */
package com.eaio.servlet;

import static org.apache.commons.lang.StringUtils.*;

import java.io.IOException;
import java.net.URL;
import java.util.regex.Pattern;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;


/**
 * Redirects all requests with a host header containing a port number except certain requests to a URL without a port but the same host and path.
 * <p>
 * Prevents search engines indexing sites at ports other than 80.
 * <p>
 * Needs an <tt>allowedURIs</tt> (note case) init parameter which is a regular expression pattern of allowed paths.
 * 
 * @author <a href="mailto:jb&#64;eaio.com">Johann Burkard</a>
 * @version $Id: RedirFilter.java 5084 2012-08-09 19:11:04Z johann $
 * @see Pattern#compile(String)
 * @see FilterConfig#getInitParameter(String)
 * @see #init(String, String)
 */
public class RedirFilter implements Filter {
    
    private final String EXAMPLE = "  <filter>\n        <filter-name>RedirFilter</filter-name>\n        <filter-class>com.eaio.servlet.RedirFilter</filter-class>\n        <init-param>\n            <param-name>allowedURIs</param-name>\n            <param-value>^/upload.*$</param-value>\n        </init-param>\n        <init-param>\n            <param-name>hostname</param-name>\n            <param-value>foo.com</param-value>\n        </init-param>\n    </filter>",
    ALLOWED_URIS = "Please define an \"allowedURIs\" init-parameter in your web.xml, like this:\n\n" + EXAMPLE + "\n\nFor now, a default value that allows all requests will be used.",
    HOSTNAME = "Please define an \"hostname\" init-parameter in your web.xml, like this:\n\n" + EXAMPLE + "\n\nFor now, a default value that allows all requests will be used.";
    
    private Pattern allowedURIs;
    
    private String hostnamePlusPort;
    
    /**
     * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
     */
    public void init(FilterConfig config) {
        String uriPattern = config.getInitParameter("allowedURIs");
        if (uriPattern == null) {
            config.getServletContext().log(ALLOWED_URIS);
            uriPattern = "^/";
        }
        String hostname = config.getInitParameter("hostname");
        if (hostname == null) {
            config.getServletContext().log(HOSTNAME);
            hostname = "guh";
        }
        init(uriPattern, hostname);
    }
    
    /**
     * For testing purposes.
     * 
     * @param uriPattern the exclusion pattern.
     */
    public void init(String uriPattern, String hostname) {
        assert uriPattern != null;
        assert hostname != null;
        allowedURIs = Pattern.compile(uriPattern);
        hostnamePlusPort = hostname + ":";
    }

    /**
     * @see javax.servlet.Filter#destroy()
     */
    public void destroy() {
    // Blank
    }

    /**
     * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse,
     * javax.servlet.FilterChain)
     */
    public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException,
            ServletException {
        HttpServletRequest request = (HttpServletRequest) req;
        HttpServletResponse response = (HttpServletResponse) res;
        
        String host = defaultString(request.getHeader("HOST"));
        String uri = defaultString(substringAfter(request.getRequestURI(), request.getContextPath()));
        String query = request.getQueryString();

        if (host.startsWith(hostnamePlusPort) && !(allowedURIs.matcher(uri).matches())) {
            URL hostURL = new URL("http://" + host);
            URL u = new URL("http", hostURL.getHost(), query == null ? uri : new StringBuilder(uri.length() + 1 + query.length()).append(uri).append('?').append(query).toString());
            response.sendRedirect(u.toString());
        }
        else {
            chain.doFilter(request, response);            
        }
       
    }

}
