/*
 * Decompiled with CFR 0.152.
 */
package net.iryndin.jdbf.writer;

import java.io.IOException;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.Arrays;
import java.util.Date;
import java.util.Map;
import net.iryndin.jdbf.core.DbfField;
import net.iryndin.jdbf.core.DbfMetadata;
import net.iryndin.jdbf.util.BitUtils;
import net.iryndin.jdbf.util.DbfMetadataUtils;
import net.iryndin.jdbf.util.JdbfUtils;

public class DbfWriter {
    private OutputStream out;
    private DbfMetadata metadata;
    private Charset stringCharset = Charset.defaultCharset();
    private byte[] recordBuffer;

    public DbfWriter(DbfMetadata metadata, OutputStream out) throws IOException {
        this.out = out;
        this.metadata = metadata;
        this.recordBuffer = new byte[metadata.getOneRecordLength()];
        this.writeHeaderAndFields();
    }

    private void writeHeaderAndFields() throws IOException {
        this.writeHeader();
        this.writeFields();
    }

    private void writeHeader() throws IOException {
        byte[] bytes = DbfMetadataUtils.toByteArrayHeader(this.metadata);
        this.out.write(bytes);
        BitUtils.memset(bytes, 0);
        this.out.write(bytes);
    }

    private void writeFields() throws IOException {
        byte[] bytes = new byte[32];
        for (DbfField f : this.metadata.getFields()) {
            DbfMetadataUtils.writeDbfField(f, bytes);
            this.out.write(bytes);
        }
        this.out.write(13);
    }

    public void write(Map<String, Object> map) throws IOException {
        BitUtils.memset(this.recordBuffer, JdbfUtils.EMPTY);
        for (DbfField f : this.metadata.getFields()) {
            Object o = map.get(f.getName());
            this.writeIntoRecordBuffer(f, o);
        }
        this.out.write(this.recordBuffer);
    }

    private void writeIntoRecordBuffer(DbfField f, Object o) {
        if (o == null) {
            return;
        }
        switch (f.getType()) {
            case Character: {
                this.writeString(f, (String)o);
                break;
            }
            case Date: {
                this.writeDate(f, (Date)o);
                break;
            }
            case Logical: {
                this.writeBoolean(f, (Boolean)o);
                break;
            }
            case Numeric: {
                this.writeBigDecimal(f, (BigDecimal)o);
                break;
            }
            case Float: {
                if (o instanceof Double) {
                    this.writeDouble(f, (Double)o);
                    break;
                }
                this.writeFloat(f, (Float)o);
                break;
            }
            case DateTime: {
                this.writeDateTime(f, (Date)o);
                break;
            }
            case Double: {
                this.writeDouble7(f, (Double)o);
                break;
            }
            case Timestamp: {
                this.writeTimestamp(f, (Date)o);
                break;
            }
            case Double7: {
                this.writeDouble7(f, (Double)o);
                break;
            }
            case Integer: {
                this.writeInteger(f, (Integer)o);
                break;
            }
            default: {
                throw new UnsupportedOperationException("Unknown or unsupported field type " + f.getType().name() + " for " + f.getName());
            }
        }
    }

    private void writeBigDecimal(DbfField f, BigDecimal value) {
        if (value != null) {
            String s = value.toPlainString();
            byte[] bytes = s.getBytes();
            if (bytes.length > f.getLength()) {
                byte[] newBytes = new byte[f.getLength()];
                System.arraycopy(bytes, 0, newBytes, 0, f.getLength());
                bytes = newBytes;
            }
            System.arraycopy(bytes, 0, this.recordBuffer, f.getOffset(), bytes.length);
        } else {
            this.blankify(f);
        }
    }

    private void writeBoolean(DbfField f, Boolean value) {
        if (value != null) {
            String s = value != false ? "T" : "F";
            byte[] bytes = s.getBytes();
            System.arraycopy(bytes, 0, this.recordBuffer, f.getOffset(), bytes.length);
        } else {
            byte[] bytes = "?".getBytes();
            System.arraycopy(bytes, 0, this.recordBuffer, f.getOffset(), bytes.length);
        }
    }

    private void writeDate(DbfField f, Date value) {
        if (value != null) {
            byte[] bytes = JdbfUtils.writeDate(value);
            System.arraycopy(bytes, 0, this.recordBuffer, f.getOffset(), bytes.length);
        } else {
            this.blankify(f);
        }
    }

    private void writeString(DbfField f, String value) {
        if (value != null) {
            byte[] bytes = value.getBytes(this.stringCharset);
            if (bytes.length > f.getLength()) {
                byte[] newBytes = new byte[f.getLength()];
                System.arraycopy(bytes, 0, newBytes, 0, f.getLength());
                bytes = newBytes;
            }
            System.arraycopy(bytes, 0, this.recordBuffer, f.getOffset(), bytes.length);
        } else {
            this.blankify(f);
        }
    }

    private void writeFloat(DbfField f, Float value) {
        this.writeDouble(f, value.doubleValue());
    }

    private void writeDouble(DbfField f, Double value) {
        if (value != null) {
            String str = String.format("% 20.18f", value);
            if (str.length() > 20) {
                str = str.substring(0, 20);
            }
            this.writeString(f, str);
        } else {
            this.blankify(f);
        }
    }

    private void writeTimestamp(DbfField f, Date d) {
        if (d != null) {
            byte[] bytes = JdbfUtils.writeJulianDate(d);
            System.arraycopy(bytes, 0, this.recordBuffer, f.getOffset(), bytes.length);
        } else {
            this.blankify(f);
        }
    }

    private void writeDateTime(DbfField f, Date d) {
        if (d != null) {
            ByteBuffer bb = ByteBuffer.allocate(8);
            bb.putLong(d.getTime());
            System.arraycopy(bb.array(), 0, this.recordBuffer, f.getOffset(), bb.capacity());
        } else {
            this.blankify(f);
        }
    }

    private void writeDouble7(DbfField f, Double d) {
        if (d != null) {
            ByteBuffer bb = ByteBuffer.allocate(8);
            bb.putDouble(d);
            System.arraycopy(bb.array(), 0, this.recordBuffer, f.getOffset(), bb.capacity());
        } else {
            this.blankify(f);
        }
    }

    private void writeInteger(DbfField f, Integer i) {
        if (i != null) {
            ByteBuffer bb = ByteBuffer.allocate(4);
            bb.putInt(i);
            System.arraycopy(bb.array(), 0, this.recordBuffer, f.getOffset(), bb.capacity());
        } else {
            this.blankify(f);
        }
    }

    private void blankify(DbfField f) {
        byte[] bytes = new byte[f.getLength()];
        Arrays.fill(bytes, (byte)32);
        System.arraycopy(bytes, 0, this.recordBuffer, f.getOffset(), bytes.length);
    }

    public void close() throws IOException {
        this.out.flush();
        this.out.close();
    }

    public void setStringCharset(String charsetName) {
        this.setStringCharset(Charset.forName(charsetName));
    }

    public void setStringCharset(Charset stringCharset) {
        this.stringCharset = stringCharset;
    }
}

