/*
 * FindBugs
 * Copyright (C) 2011 University of Maryland
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package edu.umd.cs.findbugs.annotations;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;


/**
 * Used to suppress FindBugs warnings.
 *
 * It should be used instead of
 * {@link edu.umd.cs.findbugs.annotations.SuppressWarnings} to avoid conflicts with
 * {@link java.lang.SuppressWarnings}.
 *
 */
@Retention(RetentionPolicy.CLASS)
public @interface SuppressFBWarnings {
    /**
     * The set of FindBugs warnings that are to be suppressed in
     * annotated element. The value can be a bug category, kind or pattern.
     *
     */
    String[] value() default {};

    /**
     * Optional documentation of the reason why the warning is suppressed
     */
    String justification() default "";

    /**
     * <p>By default <code>SuppressFBWarnings</code> annotations suppress bugs by prefix,
     * for instance <code> @SuppressFBWarnings(value = "EI_EXPO", justification = "It's OK")</code>
     * will suppress bugs of type <code>EI_EXPOSE_REP</code> and <code>EI_EXPOSE_REP2</code>.
     *
     * <p>You might use <code>@SuppressFBWarnings(value = "EI_EXPOSE_REP", justification = "It's OK", matchType=EXACT)</code>
     * to suppress <code>EI_EXPOSE_REP</code>, but not <code>EI_EXPOSE_REP2</code>.
     *
     * <p>Regular expressions are also supported with <code>matchType=REGEX</code>.
     *
     * @see SuppressMatchType
     */
    SuppressMatchType matchType() default SuppressMatchType.DEFAULT;
}
