/*
 * Decompiled with CFR 0.152.
 */
package com.github.nickrm.jflux;

import com.github.nickrm.jflux.AnnotationBasedPointConverter;
import com.github.nickrm.jflux.ApiCaller;
import com.github.nickrm.jflux.DatabaseManager;
import com.github.nickrm.jflux.LineProtocolConverter;
import com.github.nickrm.jflux.NamingStrategy;
import com.github.nickrm.jflux.RetentionPolicyManager;
import com.github.nickrm.jflux.api.JFluxHttpClient;
import com.github.nickrm.jflux.api.response.ResponseMetadata;
import com.github.nickrm.jflux.domain.Measurement;
import com.github.nickrm.jflux.domain.Point;
import com.github.nickrm.jflux.domain.RetentionPolicy;
import com.github.nickrm.jflux.exception.DatabaseAlreadyExistsException;
import com.github.nickrm.jflux.exception.NoDatabaseSelectedException;
import com.github.nickrm.jflux.exception.RetentionPolicyAlreadyExistsException;
import com.github.nickrm.jflux.exception.UnknownDatabaseException;
import com.github.nickrm.jflux.exception.UnknownRetentionPolicyException;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class JFluxClient
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(JFluxClient.class);
    private final JFluxHttpClient httpClient;
    private final DatabaseManager databaseManager;
    private final RetentionPolicyManager retentionPolicyManager;
    private final ApiCaller apiCaller;
    private final LineProtocolConverter lineProtocolConverter;
    private final NamingStrategy namingStrategy;
    private final AnnotationBasedPointConverter annotationBasedPointConverter;
    private String currentDatabase;

    JFluxClient(JFluxHttpClient httpClient, DatabaseManager databaseManager, RetentionPolicyManager retentionPolicyManager) throws IOException {
        this.httpClient = httpClient;
        this.databaseManager = databaseManager;
        this.retentionPolicyManager = retentionPolicyManager;
        try {
            ResponseMetadata responseMetadata = httpClient.ping();
            LOGGER.info("Connected to InfluxDB {} {} instance at {}", new Object[]{responseMetadata.getDbBuildType(), responseMetadata.getDbVersion(), httpClient.getHostUrl()});
        }
        catch (IOException e) {
            throw new IOException("Could not connect to InfluxDB instance", e);
        }
        this.apiCaller = new ApiCaller();
        this.lineProtocolConverter = new LineProtocolConverter();
        this.namingStrategy = new NamingStrategy();
        this.annotationBasedPointConverter = new AnnotationBasedPointConverter(this.namingStrategy);
    }

    public List<String> getDatabases() {
        return this.databaseManager.getDatabases();
    }

    public boolean databaseExists(String databaseName) {
        return this.databaseManager.databaseExists(databaseName);
    }

    public void createDatabase(String databaseName) {
        if (this.databaseExists(databaseName)) {
            throw new DatabaseAlreadyExistsException(databaseName);
        }
        this.databaseManager.createDatabase(databaseName);
    }

    public void dropDatabase(String databaseName) {
        if (!this.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        this.databaseManager.dropDatabase(databaseName);
    }

    public void useDatabase(String databaseName) {
        if (!this.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        this.currentDatabase = databaseName;
    }

    public List<RetentionPolicy> getRetentionPolicies() {
        this.assertDatabaseHasBeenSelected();
        return this.getRetentionPolicies(this.currentDatabase);
    }

    public List<RetentionPolicy> getRetentionPolicies(String databaseName) {
        if (!this.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        return this.retentionPolicyManager.getRetentionPolicies(databaseName);
    }

    public RetentionPolicy getRetentionPolicy(String retentionPolicyName) {
        this.assertDatabaseHasBeenSelected();
        return this.getRetentionPolicy(retentionPolicyName, this.currentDatabase);
    }

    public RetentionPolicy getRetentionPolicy(String retentionPolicyName, String databaseName) {
        if (!this.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        return this.retentionPolicyManager.getRetentionPolicy(retentionPolicyName, databaseName);
    }

    public boolean retentionPolicyExists(String retentionPolicyName) {
        this.assertDatabaseHasBeenSelected();
        return this.retentionPolicyExists(retentionPolicyName, this.currentDatabase);
    }

    public boolean retentionPolicyExists(String retentionPolicyName, String databaseName) {
        if (retentionPolicyName == null) {
            throw new IllegalArgumentException("Retention policy name cannot be null");
        }
        if (!this.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        return this.retentionPolicyManager.retentionPolicyExists(retentionPolicyName, databaseName);
    }

    public void createRetentionPolicy(RetentionPolicy retentionPolicy) {
        this.assertDatabaseHasBeenSelected();
        this.createRetentionPolicy(retentionPolicy, this.currentDatabase);
    }

    public void createRetentionPolicy(RetentionPolicy retentionPolicy, String databaseName) {
        if (!this.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        if (this.retentionPolicyExists(retentionPolicy.getName(), databaseName)) {
            throw new RetentionPolicyAlreadyExistsException(retentionPolicy.getName(), databaseName);
        }
        this.retentionPolicyManager.createRetentionPolicy(retentionPolicy, databaseName);
    }

    public void alterRetentionPolicy(String retentionPolicyName, RetentionPolicy newDefinition) {
        this.assertDatabaseHasBeenSelected();
        this.alterRetentionPolicy(retentionPolicyName, this.currentDatabase, newDefinition);
    }

    public void alterRetentionPolicy(String retentionPolicyName, String databaseName, RetentionPolicy newDefinition) {
        if (!this.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        if (!this.retentionPolicyExists(retentionPolicyName, databaseName)) {
            throw new UnknownRetentionPolicyException(retentionPolicyName, databaseName);
        }
        this.retentionPolicyManager.alterRetentionPolicy(retentionPolicyName, databaseName, newDefinition);
    }

    public void dropRetentionPolicy(String retentionPolicyName) {
        this.assertDatabaseHasBeenSelected();
        this.dropRetentionPolicy(retentionPolicyName, this.currentDatabase);
    }

    public void dropRetentionPolicy(String retentionPolicyName, String databaseName) {
        if (!this.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        if (!this.retentionPolicyExists(retentionPolicyName, databaseName)) {
            throw new UnknownRetentionPolicyException(retentionPolicyName, databaseName);
        }
        this.retentionPolicyManager.dropRetentionPolicy(retentionPolicyName, databaseName);
    }

    public void write(Object data) {
        this.write(Collections.singleton(data));
    }

    public void write(String databaseName, Object data) {
        if (data == null) {
            throw new IllegalArgumentException("Input cannot be null");
        }
        this.write(databaseName, Collections.singleton(data));
    }

    public void write(Collection<?> data) {
        this.assertDatabaseHasBeenSelected();
        this.write(this.currentDatabase, data);
    }

    public void write(String databaseName, Collection<?> data) {
        if (data.isEmpty()) {
            return;
        }
        Class<?> dataClass = data.iterator().next().getClass();
        String measurementName = this.namingStrategy.getMeasurementName(dataClass);
        List<Point> points = data.parallelStream().map(this.annotationBasedPointConverter::toPoint).collect(Collectors.toList());
        this.writePoints(databaseName, measurementName, points);
    }

    public void write(Object data, String retentionPolicyName) {
        this.write(Collections.singleton(data), retentionPolicyName);
    }

    public void write(String databaseName, Object data, String retentionPolicyName) {
        if (data == null) {
            throw new IllegalArgumentException("Input cannot be null");
        }
        this.write(databaseName, Collections.singleton(data), retentionPolicyName);
    }

    public void write(Collection<?> data, String retentionPolicyName) {
        this.assertDatabaseHasBeenSelected();
        this.write(this.currentDatabase, data, retentionPolicyName);
    }

    public void write(String databaseName, Collection<?> data, String retentionPolicyName) {
        if (data.isEmpty()) {
            return;
        }
        Class<?> dataClass = data.iterator().next().getClass();
        String measurementName = this.namingStrategy.getMeasurementName(dataClass);
        List<Point> points = data.parallelStream().map(this.annotationBasedPointConverter::toPoint).collect(Collectors.toList());
        this.writePoints(databaseName, measurementName, points, retentionPolicyName);
    }

    public void writePoint(String measurementName, Point point) {
        this.writePoints(measurementName, Collections.singleton(point));
    }

    public void writePoint(String databaseName, String measurementName, Point point) {
        this.writePoints(databaseName, measurementName, Collections.singleton(point));
    }

    public void writePoint(String measurementName, Point point, String retentionPolicyName) {
        this.writePoints(measurementName, Collections.singleton(point), retentionPolicyName);
    }

    public void writePoint(String databaseName, String measurementName, Point point, String retentionPolicyName) {
        this.writePoints(databaseName, measurementName, Collections.singleton(point), retentionPolicyName);
    }

    public void writePoints(String measurementName, Collection<Point> points) {
        this.assertDatabaseHasBeenSelected();
        this.writePoints(this.currentDatabase, measurementName, points);
    }

    public void writePoints(String databaseName, String measurementName, Collection<Point> points) {
        if (databaseName == null) {
            throw new IllegalArgumentException("Database name cannot be null");
        }
        if (measurementName == null) {
            throw new IllegalArgumentException("Measurement name cannot be null");
        }
        if (!this.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        this.lineProtocolConverter.toLineProtocol(measurementName, points).forEach(lineProtocol -> this.apiCaller.callApi(() -> this.httpClient.write(databaseName, (String)lineProtocol)));
    }

    public void writePoints(String measurementName, Collection<Point> points, String retentionPolicyName) {
        this.assertDatabaseHasBeenSelected();
        this.writePoints(this.currentDatabase, measurementName, points, retentionPolicyName);
    }

    public void writePoints(String databaseName, String measurementName, Collection<Point> points, String retentionPolicyName) {
        if (!this.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        if (!this.retentionPolicyExists(retentionPolicyName, databaseName)) {
            throw new UnknownRetentionPolicyException(retentionPolicyName, databaseName);
        }
        this.lineProtocolConverter.toLineProtocol(measurementName, points).forEach(lineProtocol -> this.apiCaller.callApi(() -> this.httpClient.write(databaseName, retentionPolicyName, (String)lineProtocol)));
    }

    public <T> List<T> getAllPoints(Class<T> targetType) {
        this.assertDatabaseHasBeenSelected();
        return this.getAllPoints(this.currentDatabase, targetType);
    }

    public <T> List<T> getAllPoints(String databaseName, Class<T> targetType) {
        String measurementName = this.namingStrategy.getMeasurementName(targetType);
        return this.getAllPoints(databaseName, measurementName).stream().map(point -> this.annotationBasedPointConverter.fromPoint((Point)point, targetType)).collect(Collectors.toList());
    }

    public List<Point> getAllPoints(String measurementName) {
        this.assertDatabaseHasBeenSelected();
        return this.getAllPoints(this.currentDatabase, measurementName);
    }

    public List<Point> getAllPoints(String databaseName, String measurementName) {
        if (measurementName == null) {
            throw new IllegalArgumentException("Measurement name cannot be blank");
        }
        if (databaseName == null) {
            throw new IllegalArgumentException("Database name cannot be null");
        }
        if (!this.databaseManager.databaseExists(databaseName)) {
            throw new UnknownDatabaseException(databaseName);
        }
        String query = "SELECT * FROM \"" + databaseName + "\"..\"" + measurementName + '\"';
        Measurement callResult = this.apiCaller.callApi(() -> this.httpClient.query(query));
        return callResult == null ? Collections.emptyList() : callResult.getPoints();
    }

    private void assertDatabaseHasBeenSelected() {
        if (this.currentDatabase == null) {
            throw new NoDatabaseSelectedException();
        }
    }

    @Override
    public void close() throws Exception {
        this.httpClient.close();
    }

    public static final class Builder {
        private String host;

        public Builder(String host) {
            this.host = host;
        }

        public JFluxClient build() throws IOException {
            JFluxHttpClient httpClient = new JFluxHttpClient.Builder(this.host).build();
            DatabaseManager databaseManager = new DatabaseManager(httpClient);
            RetentionPolicyManager retentionPolicyManager = new RetentionPolicyManager(httpClient);
            return new JFluxClient(httpClient, databaseManager, retentionPolicyManager);
        }
    }
}

