/*
 * Decompiled with CFR 0.152.
 */
package com.github.davidmoten.geo;

import com.github.davidmoten.geo.Coverage;
import com.github.davidmoten.geo.Direction;
import com.github.davidmoten.geo.LatLong;
import com.github.davidmoten.geo.Parity;
import com.github.davidmoten.geo.Position;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

public final class GeoHash {
    public static final int DEFAULT_MAX_HASHES = 12;
    private static final int[] BITS = new int[]{16, 8, 4, 2, 1};
    private static final String BASE32 = "0123456789bcdefghjkmnpqrstuvwxyz";
    private static final Map<Direction, Map<Parity, String>> NEIGHBOURS = GeoHash.createNeighbours();
    private static final Map<Direction, Map<Parity, String>> BORDERS = GeoHash.createBorders();
    public static final int MAX_HASH_LENGTH = 12;
    private static Double[] hashHeightCache = new Double[12];
    private static Double[] hashWidthCache = new Double[12];

    private GeoHash() {
    }

    private static Map<Direction, Map<Parity, String>> createBorders() {
        Map<Direction, Map<Parity, String>> m = GeoHash.createDirectionParityMap();
        m.get((Object)Direction.RIGHT).put(Parity.EVEN, "bcfguvyz");
        m.get((Object)Direction.LEFT).put(Parity.EVEN, "0145hjnp");
        m.get((Object)Direction.TOP).put(Parity.EVEN, "prxz");
        m.get((Object)Direction.BOTTOM).put(Parity.EVEN, "028b");
        GeoHash.addOddParityEntries(m);
        return m;
    }

    private static Map<Direction, Map<Parity, String>> createNeighbours() {
        Map<Direction, Map<Parity, String>> m = GeoHash.createDirectionParityMap();
        m.get((Object)Direction.RIGHT).put(Parity.EVEN, "bc01fg45238967deuvhjyznpkmstqrwx");
        m.get((Object)Direction.LEFT).put(Parity.EVEN, "238967debc01fg45kmstqrwxuvhjyznp");
        m.get((Object)Direction.TOP).put(Parity.EVEN, "p0r21436x8zb9dcf5h7kjnmqesgutwvy");
        m.get((Object)Direction.BOTTOM).put(Parity.EVEN, "14365h7k9dcfesgujnmqp0r2twvyx8zb");
        GeoHash.addOddParityEntries(m);
        return m;
    }

    private static Map<Direction, Map<Parity, String>> createDirectionParityMap() {
        HashMap m = Maps.newHashMap();
        m.put(Direction.BOTTOM, Maps.newHashMap());
        m.put(Direction.TOP, Maps.newHashMap());
        m.put(Direction.LEFT, Maps.newHashMap());
        m.put(Direction.RIGHT, Maps.newHashMap());
        return m;
    }

    private static void addOddParityEntries(Map<Direction, Map<Parity, String>> m) {
        m.get((Object)Direction.BOTTOM).put(Parity.ODD, m.get((Object)Direction.LEFT).get((Object)Parity.EVEN));
        m.get((Object)Direction.TOP).put(Parity.ODD, m.get((Object)Direction.RIGHT).get((Object)Parity.EVEN));
        m.get((Object)Direction.LEFT).put(Parity.ODD, m.get((Object)Direction.BOTTOM).get((Object)Parity.EVEN));
        m.get((Object)Direction.RIGHT).put(Parity.ODD, m.get((Object)Direction.TOP).get((Object)Parity.EVEN));
    }

    public static String adjacentHash(String hash, Direction direction) {
        GeoHash.checkHash(hash);
        String source = hash.toLowerCase();
        char lastChar = source.charAt(source.length() - 1);
        Parity parity = source.length() % 2 == 0 ? Parity.EVEN : Parity.ODD;
        String base = source.substring(0, source.length() - 1);
        if (BORDERS.get((Object)direction).get((Object)parity).indexOf(lastChar) != -1) {
            base = GeoHash.adjacentHash(base, direction);
        }
        return base + BASE32.charAt(NEIGHBOURS.get((Object)direction).get((Object)parity).indexOf(lastChar));
    }

    private static void checkHash(String hash) {
        Preconditions.checkArgument((hash != null && hash.length() > 0 ? 1 : 0) != 0, (Object)"hash must be non-null of length>1");
    }

    public static String right(String hash) {
        return GeoHash.adjacentHash(hash, Direction.RIGHT);
    }

    public static String left(String hash) {
        return GeoHash.adjacentHash(hash, Direction.LEFT);
    }

    public static String top(String hash) {
        return GeoHash.adjacentHash(hash, Direction.TOP);
    }

    public static String bottom(String hash) {
        return GeoHash.adjacentHash(hash, Direction.BOTTOM);
    }

    public static String adjacentHash(String hash, Direction direction, int steps) {
        if (steps < 0) {
            return GeoHash.adjacentHash(hash, direction.opposite(), Math.abs(steps));
        }
        String h = hash;
        for (int i = 0; i < steps; ++i) {
            h = GeoHash.adjacentHash(h, direction);
        }
        return h;
    }

    public static List<String> neighbours(String hash) {
        ArrayList list = Lists.newArrayList();
        String left = GeoHash.adjacentHash(hash, Direction.LEFT);
        String right = GeoHash.adjacentHash(hash, Direction.RIGHT);
        list.add(left);
        list.add(right);
        list.add(GeoHash.adjacentHash(hash, Direction.TOP));
        list.add(GeoHash.adjacentHash(hash, Direction.BOTTOM));
        list.add(GeoHash.adjacentHash(left, Direction.TOP));
        list.add(GeoHash.adjacentHash(left, Direction.BOTTOM));
        list.add(GeoHash.adjacentHash(right, Direction.TOP));
        list.add(GeoHash.adjacentHash(right, Direction.BOTTOM));
        return list;
    }

    public static String encodeHash(double latitude, double longitude) {
        return GeoHash.encodeHash(latitude, longitude, 12);
    }

    public static String encodeHash(LatLong p, int length) {
        return GeoHash.encodeHash(p.getLat(), p.getLon(), length);
    }

    public static String encodeHash(double latitude, double longitude, int length) {
        Preconditions.checkArgument((length > 0 ? 1 : 0) != 0, (Object)"length must be greater than zero");
        Preconditions.checkArgument((latitude >= -90.0 && latitude <= 90.0 ? 1 : 0) != 0, (Object)"latitude must be between -90 and 90 inclusive");
        longitude = Position.to180(longitude);
        boolean isEven = true;
        double[] lat = new double[2];
        double[] lon = new double[2];
        int bit = 0;
        int ch = 0;
        StringBuilder geohash = new StringBuilder();
        lat[0] = -90.0;
        lat[1] = 90.0;
        lon[0] = -180.0;
        lon[1] = 180.0;
        while (geohash.length() < length) {
            double mid;
            if (isEven) {
                mid = (lon[0] + lon[1]) / 2.0;
                if (longitude > mid) {
                    ch |= BITS[bit];
                    lon[0] = mid;
                } else {
                    lon[1] = mid;
                }
            } else {
                mid = (lat[0] + lat[1]) / 2.0;
                if (latitude > mid) {
                    ch |= BITS[bit];
                    lat[0] = mid;
                } else {
                    lat[1] = mid;
                }
            }
            boolean bl = isEven = !isEven;
            if (bit < 4) {
                ++bit;
                continue;
            }
            geohash.append(BASE32.charAt(ch));
            bit = 0;
            ch = 0;
        }
        return geohash.toString();
    }

    public static LatLong decodeHash(String geohash) {
        Preconditions.checkNotNull((Object)geohash, (Object)"geohash cannot be null");
        boolean isEven = true;
        double[] lat = new double[2];
        double[] lon = new double[2];
        lat[0] = -90.0;
        lat[1] = 90.0;
        lon[0] = -180.0;
        lon[1] = 180.0;
        for (int i = 0; i < geohash.length(); ++i) {
            char c = geohash.charAt(i);
            int cd = BASE32.indexOf(c);
            for (int j = 0; j < 5; ++j) {
                int mask = BITS[j];
                if (isEven) {
                    GeoHash.refineInterval(lon, cd, mask);
                } else {
                    GeoHash.refineInterval(lat, cd, mask);
                }
                isEven = !isEven;
            }
        }
        double resultLat = (lat[0] + lat[1]) / 2.0;
        double resultLon = (lon[0] + lon[1]) / 2.0;
        return new LatLong(resultLat, resultLon);
    }

    private static void refineInterval(double[] interval, int cd, int mask) {
        if ((cd & mask) != 0) {
            interval[0] = (interval[0] + interval[1]) / 2.0;
        } else {
            interval[1] = (interval[0] + interval[1]) / 2.0;
        }
    }

    public static int hashLengthToCoverBoundingBox(double topLeftLat, double topLeftLon, double bottomRightLat, double bottomRightLon) {
        for (int i = 12; i >= 1; --i) {
            String hash = GeoHash.encodeHash(topLeftLat, topLeftLon, i);
            if (!GeoHash.hashContains(hash, bottomRightLat, bottomRightLon)) continue;
            return i;
        }
        return 0;
    }

    public static boolean hashContains(String hash, double lat, double lon) {
        LatLong centre = GeoHash.decodeHash(hash);
        return Math.abs(centre.getLat() - lat) <= GeoHash.heightDegrees(hash.length()) / 2.0 && Math.abs(Position.to180(centre.getLon() - lon)) <= GeoHash.widthDegrees(hash.length()) / 2.0;
    }

    public static Coverage coverBoundingBox(double topLeftLat, double topLeftLon, double bottomRightLat, double bottomRightLon) {
        return GeoHash.coverBoundingBoxMaxHashes(topLeftLat, topLeftLon, bottomRightLat, bottomRightLon, 12);
    }

    public static Coverage coverBoundingBoxMaxHashes(double topLeftLat, double topLeftLon, double bottomRightLat, double bottomRightLon, int maxHashes) {
        int startLength;
        Coverage coverage = null;
        for (int length = startLength = GeoHash.hashLengthToCoverBoundingBox(topLeftLat, topLeftLon, bottomRightLat, bottomRightLon); length <= 12; ++length) {
            Coverage c = GeoHash.coverBoundingBox(topLeftLat, topLeftLon, bottomRightLat, bottomRightLon, length);
            if (c.getHashes().size() > maxHashes) {
                return coverage;
            }
            coverage = c;
        }
        return coverage;
    }

    public static Coverage coverBoundingBox(double topLeftLat, double topLeftLon, double bottomRightLat, double bottomRightLon, int length) {
        double lat;
        Preconditions.checkArgument((length > 0 ? 1 : 0) != 0, (Object)"length must be greater than zero");
        double actualWidthDegreesPerHash = GeoHash.widthDegrees(length);
        double actualHeightDegreesPerHash = GeoHash.heightDegrees(length);
        TreeSet hashes = Sets.newTreeSet();
        double diff = Position.longitudeDiff(bottomRightLon, topLeftLon);
        double maxLon = topLeftLon + diff;
        for (lat = bottomRightLat; lat <= topLeftLat; lat += actualHeightDegreesPerHash) {
            for (double lon = topLeftLon; lon <= maxLon; lon += actualWidthDegreesPerHash) {
                GeoHash.addHash(hashes, lat, lon, length);
            }
        }
        for (lat = bottomRightLat; lat <= topLeftLat; lat += actualHeightDegreesPerHash) {
            GeoHash.addHash(hashes, lat, maxLon, length);
        }
        for (double lon = topLeftLon; lon <= maxLon; lon += actualWidthDegreesPerHash) {
            GeoHash.addHash(hashes, topLeftLat, lon, length);
        }
        GeoHash.addHash(hashes, topLeftLat, maxLon, length);
        double areaDegrees = diff * (topLeftLat - bottomRightLat);
        double coverageAreaDegrees = (double)hashes.size() * GeoHash.widthDegrees(length) * GeoHash.heightDegrees(length);
        double ratio = coverageAreaDegrees / areaDegrees;
        return new Coverage(hashes, ratio);
    }

    private static void addHash(Set<String> hashes, double lat, double lon, int length) {
        hashes.add(GeoHash.encodeHash(lat, lon, length));
    }

    public static synchronized double heightDegrees(int n) {
        if (n > 0 && n <= 12) {
            if (hashHeightCache[n - 1] == null) {
                GeoHash.hashHeightCache[n - 1] = GeoHash.calculateHeightDegrees(n);
            }
            return hashHeightCache[n - 1];
        }
        return GeoHash.calculateHeightDegrees(n);
    }

    private static double calculateHeightDegrees(int n) {
        double a = n % 2 == 0 ? 0.0 : -0.5;
        double result = 180.0 / Math.pow(2.0, 2.5 * (double)n + a);
        return result;
    }

    public static synchronized double widthDegrees(int n) {
        if (n > 0 && n <= 12) {
            if (hashWidthCache[n - 1] == null) {
                GeoHash.hashWidthCache[n - 1] = GeoHash.calculateWidthDegrees(n);
            }
            return hashWidthCache[n - 1];
        }
        return GeoHash.calculateWidthDegrees(n);
    }

    private static double calculateWidthDegrees(int n) {
        double a = n % 2 == 0 ? -1.0 : -0.5;
        double result = 180.0 / Math.pow(2.0, 2.5 * (double)n + a);
        return result;
    }

    public static String gridAsString(String hash, int size, Set<String> highlightThese) {
        return GeoHash.gridAsString(hash, -size, -size, size, size, highlightThese);
    }

    public static String gridAsString(String hash, int fromRight, int fromBottom, int toRight, int toBottom) {
        return GeoHash.gridAsString(hash, fromRight, fromBottom, toRight, toBottom, Collections.<String>emptySet());
    }

    public static String gridAsString(String hash, int fromRight, int fromBottom, int toRight, int toBottom, Set<String> highlightThese) {
        StringBuilder s = new StringBuilder();
        for (int bottom = fromBottom; bottom <= toBottom; ++bottom) {
            for (int right = fromRight; right <= toRight; ++right) {
                String h = GeoHash.adjacentHash(hash, Direction.RIGHT, right);
                if (highlightThese.contains(h = GeoHash.adjacentHash(h, Direction.BOTTOM, bottom))) {
                    h = h.toUpperCase();
                }
                s.append(h).append(" ");
            }
            s.append("\n");
        }
        return s.toString();
    }
}

