package com.crabshue.commons.validations.path.validators;

import java.nio.file.Files;
import java.nio.file.Path;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import com.crabshue.commons.validations.path.annotations.PathAccessMode;
import com.crabshue.commons.validations.path.annotations.ValidDirectoryPath;

public class DirectoryPathValidator implements ConstraintValidator<ValidDirectoryPath, Path> {

    private PathAccessMode[] accessModes;

    @Override
    public void initialize(ValidDirectoryPath constraintAnnotation) {
        accessModes = constraintAnnotation.accessModes();
    }

    @Override
    public boolean isValid(Path path, ConstraintValidatorContext constraintValidatorContext) {

        if (path != null) {
            if (!Files.exists(path)) {
                setErrorMessage(constraintValidatorContext, "The directory '${validatedValue}' does not exist");
                return false;
            }

            if (!Files.isDirectory(path)) {
                setErrorMessage(constraintValidatorContext, "'${validatedValue}' is not a directory");
                return false;
            }

            for (PathAccessMode accessMode : accessModes) {
                if (!accessMode.canAccessInMode(path)) {
                    setErrorMessage(constraintValidatorContext, "The directory '${validatedValue}' cannot be accessed in " + accessMode.getAccessMode());
                    return false;
                }
            }
        }

        return true;
    }

    private void setErrorMessage(ConstraintValidatorContext constraintValidatorContext, String message) {
        constraintValidatorContext.disableDefaultConstraintViolation();

        constraintValidatorContext
            .buildConstraintViolationWithTemplate(message)
            .addConstraintViolation();
    }
}
