package com.flybits.commons.library.encryption

import android.annotation.TargetApi
import android.content.Context
import android.content.SharedPreferences
import android.os.Build
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyProperties
import android.util.Base64
import android.util.Log
import com.flybits.commons.library.encryption.SecuredPreferenceStore.KeyStoreRecoveryNotifier
import com.flybits.commons.library.logging.Logger
import java.io.*
import java.math.BigInteger
import java.security.*
import java.security.cert.CertificateException
import java.security.interfaces.RSAPrivateKey
import java.security.interfaces.RSAPublicKey
import java.security.spec.MGF1ParameterSpec
import java.util.*
import javax.crypto.*
import javax.crypto.spec.GCMParameterSpec
import javax.crypto.spec.OAEPParameterSpec
import javax.crypto.spec.PSource
import javax.security.auth.x500.X500Principal


@Suppress("DEPRECATION")
/**
 * This class is the helper class that alow you to utilize encrypt/decrypt method
 * Most encryption logic stay here
 * @param context application context
 * @param prefStore backing store for storing information
 * @param keyAliasPrefix prefix for key aliases
 * @param bitShiftingKey a key to use for randomization (seed) and bit shifting, this enhances
 * the security on older OS versions a bit
 * @param recoveryHandler callback/listener for recovery notification
 * @throws IOException
 * @throws NoSuchAlgorithmException
 * @throws InvalidAlgorithmParameterException
 * @throws NoSuchProviderException
 * @throws NoSuchPaddingException
 * @throws CertificateException
 * @throws KeyStoreException
 * @throws UnrecoverableEntryException
 * @throws InvalidKeyException
 * @throws IllegalStateException
 */
class EncryptionManager(
    context: Context?,
    prefStore: SharedPreferences,
    keyAliasPrefix: String?,
    bitShiftingKey: ByteArray?,
    recoveryHandler: KeyStoreRecoveryNotifier?
) {
    private val AES_BIT_LENGTH = 256
    private val GCM_TAG_LENGTH = 128
    private val IV_LENGTH = 12
    private val KEYSTORE_PROVIDER = "AndroidKeyStore"
    private val SHIFTING_KEY: ByteArray?
    private val RSA_KEY_ALIAS: String
    val AES_KEY_ALIAS: String
    val MAC_KEY_ALIAS: String
    private val DELIMITER = "]"
    private val AES_CIPHER = KEY_ALGORITHM_AES + "/" +
            BLOCK_MODE_GCM + "/" +
            ENCRYPTION_PADDING_NONE
    val IS_COMPAT_MODE_KEY_ALIAS: String
    private lateinit var mStore: KeyStore
    private var aesKey: SecretKey? = null
    private val mKeyAliasPrefix: String
    private val mContext: Context?
    var mPrefs: SharedPreferences
    var mRecoveryHandler: KeyStoreRecoveryNotifier?

    //    private final String x500_PRINCIPAL = "CN = Secured Preference Store, O = Devliving Online";
    private val x500_PRINCIPAL = "CN = Flybits, O = Flybits Mobile"

    /**
     * @param context
     * @param prefStore
     * @param recoveryNotifier
     * @throws IOException
     * @throws CertificateException
     * @throws NoSuchAlgorithmException
     * @throws KeyStoreException
     * @throws UnrecoverableEntryException
     * @throws InvalidAlgorithmParameterException
     * @throws NoSuchPaddingException
     * @throws InvalidKeyException
     * @throws NoSuchProviderException
     */
    @Deprecated(
        """Use the full constructor for better security on older versions of Android
      """
    )
    constructor(
        context: Context?,
        prefStore: SharedPreferences,
        recoveryNotifier: KeyStoreRecoveryNotifier?
    ) : this(context, prefStore, null, null, recoveryNotifier) {
    }

    private fun <T : Exception?> isRecoverableError(error: T): Boolean {
        return (error is KeyStoreException
                || error is UnrecoverableEntryException
                || error is InvalidKeyException
                || error is IllegalStateException
                || error is IOException && error.cause != null && error.cause is BadPaddingException)
    }

    @Throws(
        NoSuchPaddingException::class,
        InvalidKeyException::class,
        NoSuchAlgorithmException::class,
        KeyStoreException::class,
        UnrecoverableEntryException::class,
        NoSuchProviderException::class,
        InvalidAlgorithmParameterException::class,
        IOException::class
    )
    private fun setup(
        context: Context?,
        prefStore: SharedPreferences,
        seed: ByteArray?
    ) {
        val keyGenerated = generateKey(context, seed, prefStore)
        if (keyGenerated) {
            //store the alias prefix
            mPrefs?.edit()?.putString(
                getHashed(OVERRIDING_KEY_ALIAS_PREFIX_NAME),
                mKeyAliasPrefix
            )?.commit()
        }
        loadKey(prefStore)
    }

    private fun <T : Exception> tryRecovery(e: T): Boolean {
        return mRecoveryHandler != null && mRecoveryHandler!!.onRecoveryRequired(
            e,
            mStore,
            keyAliases()
        )
    }

    private fun keyAliases(): List<String?> {
        return Arrays.asList(AES_KEY_ALIAS, RSA_KEY_ALIAS)
    }

    /**
     * Tries to recover once if a Keystore error occurs
     * @param bytes
     * @return
     * @throws NoSuchPaddingException
     * @throws InvalidAlgorithmParameterException
     * @throws NoSuchAlgorithmException
     * @throws IOException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     */
    @Throws(
        NoSuchPaddingException::class,
        InvalidAlgorithmParameterException::class,
        NoSuchAlgorithmException::class,
        IOException::class,
        BadPaddingException::class,
        IllegalBlockSizeException::class,
        NoSuchProviderException::class,
        InvalidKeyException::class,
        KeyStoreException::class,
        UnrecoverableEntryException::class
    )
    @JvmSynthetic
    internal fun tryEncrypt(bytes: ByteArray?): EncryptedData? {
        var result: EncryptedData? = null
        var tryAgain = false
        try {
            result = encrypt(bytes)
        } catch (ex: Exception) {
            ex.printStackTrace()
            tryAgain = if (isRecoverableError(ex)) tryRecovery(ex) else throw ex
        }
        if (tryAgain) {
            setup(mContext, mPrefs, null)
            result = encrypt(bytes)
        }
        return result
    }

    /**
     * tries recovery once if a Keystore error occurs
     * @param data
     * @return
     * @throws NoSuchPaddingException
     * @throws InvalidAlgorithmParameterException
     * @throws NoSuchAlgorithmException
     * @throws KeyStoreException
     * @throws UnrecoverableEntryException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     * @throws IOException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     * @throws InvalidMacException
     */
    @Throws(
        NoSuchPaddingException::class,
        InvalidAlgorithmParameterException::class,
        NoSuchAlgorithmException::class,
        KeyStoreException::class,
        UnrecoverableEntryException::class,
        NoSuchProviderException::class,
        InvalidKeyException::class,
        IOException::class,
        BadPaddingException::class,
        IllegalBlockSizeException::class,
        InvalidMacException::class
    )

    @JvmSynthetic
    internal fun tryDecrypt(data: EncryptedData?): ByteArray? {
        var result: ByteArray? = null
        var tryAgain = false
        try {
            result = decrypt(data)
        } catch (ex: Exception) {
            ex.printStackTrace()
            tryAgain = if (isRecoverableError(ex)) tryRecovery(ex) else throw ex
        }

        try {
            if (tryAgain) {
                setup(mContext, mPrefs, null)
                result = decrypt(data)
            }
        } catch (ex: Exception) {
            ex.printStackTrace()
        }
        return result
    }

    /**
     * @param bytes
     * @return
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     * @throws IOException
     * @throws BadPaddingException
     * @throws NoSuchProviderException
     * @throws IllegalBlockSizeException
     * @throws InvalidAlgorithmParameterException
     */
    @Throws(
        NoSuchPaddingException::class,
        NoSuchAlgorithmException::class,
        InvalidKeyException::class,
        IOException::class,
        BadPaddingException::class,
        NoSuchProviderException::class,
        IllegalBlockSizeException::class,
        InvalidAlgorithmParameterException::class
    )

    @JvmSynthetic
    internal fun encrypt(bytes: ByteArray?): EncryptedData? {
        if (bytes != null && bytes.size > 0) {
            val IV = iV
            return encryptAES(bytes, IV)
        }
        return null
    }

    /**
     *
     * @param data
     * @return
     * @throws IOException
     * @throws NoSuchPaddingException
     * @throws InvalidAlgorithmParameterException
     * @throws NoSuchAlgorithmException
     * @throws IllegalBlockSizeException
     * @throws BadPaddingException
     * @throws InvalidMacException
     * @throws NoSuchProviderException
     * @throws InvalidKeyException
     */
    @Throws(
        IOException::class,
        NoSuchPaddingException::class,
        InvalidAlgorithmParameterException::class,
        NoSuchAlgorithmException::class,
        IllegalBlockSizeException::class,
        BadPaddingException::class,
        InvalidMacException::class,
        NoSuchProviderException::class,
        InvalidKeyException::class
    )

    @JvmSynthetic
    internal fun decrypt(data: EncryptedData?): ByteArray? {
        return if (data?.encryptedData != null) {
            decryptAES(data)
        } else null
    }

    /**
     *
     * @param text
     * @return base64 encoded encrypted data
     * @throws InvalidKeyException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchPaddingException
     * @throws IOException
     * @throws IllegalBlockSizeException
     * @throws InvalidAlgorithmParameterException
     * @throws NoSuchProviderException
     * @throws BadPaddingException
     */
    @Throws(
        InvalidKeyException::class,
        NoSuchAlgorithmException::class,
        NoSuchPaddingException::class,
        IOException::class,
        IllegalBlockSizeException::class,
        InvalidAlgorithmParameterException::class,
        NoSuchProviderException::class,
        BadPaddingException::class,
        KeyStoreException::class,
        UnrecoverableEntryException::class
    )

    @JvmSynthetic
    internal fun encrypt(text: String?): String? {
        if (text != null && text.length > 0) {
            val encrypted =
                tryEncrypt(text.toByteArray(charset(DEFAULT_CHARSET))) ?: return null
            return encodeEncryptedData(encrypted)
        }
        return null
    }

    /**
     *
     * @param text base64 encoded encrypted data
     * @return
     * @throws InvalidKeyException
     * @throws NoSuchAlgorithmException
     * @throws NoSuchPaddingException
     * @throws IOException
     * @throws IllegalBlockSizeException
     * @throws InvalidAlgorithmParameterException
     * @throws NoSuchProviderException
     * @throws BadPaddingException
     */
    @Throws(
        IOException::class,
        NoSuchPaddingException::class,
        InvalidKeyException::class,
        NoSuchAlgorithmException::class,
        IllegalBlockSizeException::class,
        BadPaddingException::class,
        InvalidMacException::class,
        NoSuchProviderException::class,
        InvalidAlgorithmParameterException::class,
        KeyStoreException::class,
        UnrecoverableEntryException::class
    )

    @JvmSynthetic
    internal fun decrypt(text: String?): String? {
        if (text != null && text.isNotEmpty()) {
            val encryptedData =
                decodeEncryptedText(text)
            val decrypted = tryDecrypt(encryptedData) ?: return null
            return String(decrypted, 0, decrypted.size, charset(DEFAULT_CHARSET))
        }

        return null
    }

    /**
     *
     * @param fileIn encrypted file
     * @param fileOut file to store decrypted data
     * @throws IOException
     * @throws NoSuchProviderException
     * @throws InvalidAlgorithmParameterException
     * @throws NoSuchAlgorithmException
     * @throws InvalidKeyException
     * @throws NoSuchPaddingException
     */
    @Throws(
        IOException::class,
        NoSuchProviderException::class,
        InvalidAlgorithmParameterException::class,
        NoSuchAlgorithmException::class,
        InvalidKeyException::class,
        NoSuchPaddingException::class
    )

    private fun encodeEncryptedData(data: EncryptedData): String {
        return if (data.mac != null) {
            base64Encode(data.IV) + DELIMITER + base64Encode(
                data.encryptedData
            ) + DELIMITER + base64Encode(data.mac)
        } else {
            base64Encode(data.IV) + DELIMITER + base64Encode(
                data.encryptedData
            )
        }
    }

    private fun decodeEncryptedText(text: String): EncryptedData {
        val result =
            EncryptedData()
        val parts = text.split(DELIMITER.toRegex()).toTypedArray()
        result.IV = base64Decode(parts[0])
        result.encryptedData = base64Decode(parts[1])
        if (parts.size > 2) {
            result.mac = base64Decode(parts[2])
        }
        return result
    }

    @Throws(
        KeyStoreException::class,
        CertificateException::class,
        NoSuchAlgorithmException::class,
        IOException::class
    )

    private fun loadKeyStore() {
        mStore = KeyStore.getInstance(KEYSTORE_PROVIDER)
        Logger.appendTag(TAG).d("loadKeyStore")
        mStore.load(null)
    }

    @get:Throws(UnsupportedEncodingException::class)
    val iV: ByteArray
        get() {
            val iv = ByteArray(IV_LENGTH)
            val rng = SecureRandom()
            rng.nextBytes(iv)
            return iv
        }

    /**
     *
     * @param IV Initialisation Vector
     * @param modeEncrypt if true then cipher is for encryption, decryption otherwise
     * @return a Cipher
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws InvalidAlgorithmParameterException
     * @throws InvalidKeyException
     */
    @TargetApi(Build.VERSION_CODES.KITKAT)
    @Throws(
        NoSuchPaddingException::class,
        NoSuchAlgorithmException::class,
        InvalidAlgorithmParameterException::class,
        InvalidKeyException::class
    )
    private fun getCipherAES(IV: ByteArray?, modeEncrypt: Boolean): Cipher {
        val cipher = Cipher.getInstance(AES_CIPHER).apply {
            init(if (modeEncrypt) Cipher.ENCRYPT_MODE else Cipher.DECRYPT_MODE, aesKey,
                GCMParameterSpec(GCM_TAG_LENGTH, IV)
            )
        }
        return cipher
    }

    /**
     *
     * @param bytes
     * @param IV
     * @return IV and Encrypted data
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws InvalidAlgorithmParameterException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     * @throws UnsupportedEncodingException
     */
    @TargetApi(Build.VERSION_CODES.KITKAT)
    @Throws(
        NoSuchPaddingException::class,
        NoSuchAlgorithmException::class,
        InvalidAlgorithmParameterException::class,
        InvalidKeyException::class,
        BadPaddingException::class,
        IllegalBlockSizeException::class,
        UnsupportedEncodingException::class
    )
    @JvmSynthetic
    internal fun encryptAES(
        bytes: ByteArray?,
        IV: ByteArray?
    ): EncryptedData? {
        var result:EncryptedData? = null
        try {
            val cipher = getCipherAES(IV, true)
             result =
                EncryptedData()
            result.IV = cipher.iv
            result.encryptedData = cipher.doFinal(bytes)
        } catch (e: Exception) {
            e.printStackTrace()
        }

        return result
    }

    /**
     *
     * @param encryptedData - IV and Encrypted data
     * @return decrypted data
     * @throws NoSuchPaddingException
     * @throws NoSuchAlgorithmException
     * @throws InvalidAlgorithmParameterException
     * @throws InvalidKeyException
     * @throws BadPaddingException
     * @throws IllegalBlockSizeException
     * @throws UnsupportedEncodingException
     */
    @TargetApi(Build.VERSION_CODES.KITKAT)
    @Throws(
        NoSuchPaddingException::class,
        NoSuchAlgorithmException::class,
        InvalidAlgorithmParameterException::class,
        InvalidKeyException::class,
        BadPaddingException::class,
        IllegalBlockSizeException::class,
        UnsupportedEncodingException::class
    )
    @JvmSynthetic
    internal fun decryptAES(encryptedData: EncryptedData): ByteArray {
        val cipher = getCipherAES(encryptedData.IV, false)
        return cipher.doFinal(encryptedData.encryptedData)
    }

    @Throws(
        KeyStoreException::class,
        UnrecoverableEntryException::class,
        NoSuchAlgorithmException::class,
        NoSuchPaddingException::class,
        NoSuchProviderException::class,
        InvalidKeyException::class,
        IOException::class
    )
    @JvmSynthetic
    internal fun loadKey(prefStore: SharedPreferences) {
        Logger.appendTag(TAG).d("loadKey")

        if (mStore.containsAlias(AES_KEY_ALIAS)) {
            val entry =
                mStore.getEntry(AES_KEY_ALIAS, null) as KeyStore.SecretKeyEntry
            aesKey = entry.secretKey
        }
    }

    @Throws(
        KeyStoreException::class,
        NoSuchProviderException::class,
        NoSuchAlgorithmException::class,
        InvalidAlgorithmParameterException::class,
        UnrecoverableEntryException::class,
        NoSuchPaddingException::class,
        InvalidKeyException::class,
        IOException::class
    )
    @JvmSynthetic
    internal fun generateKey(
        context: Context?,
        seed: ByteArray?,
        prefStore: SharedPreferences
    ): Boolean {
        var keyGenerated = false
        keyGenerated = generateAESKey(seed)
        return keyGenerated
    }

    @TargetApi(Build.VERSION_CODES.M)
    @Throws(
        KeyStoreException::class,
        NoSuchProviderException::class,
        NoSuchAlgorithmException::class,
        InvalidAlgorithmParameterException::class
    )
    private fun generateAESKey(seed: ByteArray?): Boolean {
        if (!mStore.containsAlias(AES_KEY_ALIAS)) {
            val keyGen = KeyGenerator.getInstance(
                KeyProperties.KEY_ALGORITHM_AES,
                KEYSTORE_PROVIDER
            )
            Logger.appendTag(TAG).d("creating a keystore")
            val spec = KeyGenParameterSpec.Builder(
                AES_KEY_ALIAS,
                KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT
            )
                .setCertificateSubject(X500Principal(x500_PRINCIPAL))
                .setCertificateSerialNumber(BigInteger.ONE)
                .setKeySize(AES_BIT_LENGTH)
                .setBlockModes(KeyProperties.BLOCK_MODE_GCM)
                .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_NONE)
                .setRandomizedEncryptionRequired(false) //TODO: set to true and let the Cipher generate a secured IV
                .build()
            if (seed != null && seed.isNotEmpty()) {
                val random = SecureRandom(seed)
                keyGen.init(spec, random)
            } else {
                keyGen.init(spec)
            }
            keyGen.generateKey()
            return true
        }
        return false
    }

    class EncryptedData {
        var IV: ByteArray?
        var encryptedData: ByteArray?
        var mac: ByteArray?

        constructor() {
            IV = null
            encryptedData = null
            mac = null
        }

        constructor(IV: ByteArray?, encryptedData: ByteArray?, mac: ByteArray?) {
            this.IV = IV
            this.encryptedData = encryptedData
            this.mac = mac
        }

    }

    inner class InvalidMacException :
        GeneralSecurityException("Invalid Mac, failed to verify integrity.")

    companion object {
        private const val DEFAULT_CHARSET = "UTF-8"
        private const val RSA_KEY_ALIAS_NAME = "rsa_key"
        private const val AES_KEY_ALIAS_NAME = "aes_key"
        private const val MAC_KEY_ALIAS_NAME = "mac_key"
        const val OVERRIDING_KEY_ALIAS_PREFIX_NAME = "OverridingAlias"
        const val DEFAULT_KEY_ALIAS_PREFIX = "sps"
        private const val KEY_ALGORITHM_AES = "AES"
        private const val BLOCK_MODE_GCM = "GCM"
        private const val ENCRYPTION_PADDING_NONE = "NoPadding"
        private const val IS_COMPAT_MODE_KEY_ALIAS_NAME = "data_in_compat"

        @Throws(
            NoSuchAlgorithmException::class,
            UnsupportedEncodingException::class
        )
        fun getHashed(text: String?): String {
            if (text.isNullOrBlank()) return ""
            val digest =
                MessageDigest.getInstance("SHA-256")
            val result =
                digest.digest(text.toByteArray(charset(DEFAULT_CHARSET)))
            return toHex(result)
        }

        fun toHex(data: ByteArray): String {
            val sb = StringBuilder()
            for (b in data) {
                sb.append(String.format("%02X", b))
            }
            return sb.toString()
        }

        fun base64Encode(data: ByteArray?): String {
            return Base64.encodeToString(data, Base64.NO_WRAP)
        }

        fun base64Decode(text: String?): ByteArray {
            return Base64.decode(text, Base64.NO_WRAP)
        }
    }

    init {
        var keyAliasPrefix = keyAliasPrefix
        SHIFTING_KEY = bitShiftingKey
        keyAliasPrefix = prefStore.getString(
            getHashed(OVERRIDING_KEY_ALIAS_PREFIX_NAME),
            keyAliasPrefix
        )
        mKeyAliasPrefix =
            keyAliasPrefix ?: DEFAULT_KEY_ALIAS_PREFIX
        IS_COMPAT_MODE_KEY_ALIAS = String.format(
            "%s_%s",
            mKeyAliasPrefix,
            IS_COMPAT_MODE_KEY_ALIAS_NAME
        )
        RSA_KEY_ALIAS = String.format(
            "%s_%s",
            mKeyAliasPrefix,
            RSA_KEY_ALIAS_NAME
        )
        AES_KEY_ALIAS = String.format(
            "%s_%s",
            mKeyAliasPrefix,
            AES_KEY_ALIAS_NAME
        )
        MAC_KEY_ALIAS = String.format(
            "%s_%s",
            mKeyAliasPrefix,
            MAC_KEY_ALIAS_NAME
        )
        mRecoveryHandler = recoveryHandler
        mContext = context
        mPrefs = prefStore
        loadKeyStore()
        var tryAgain = false
        try {
            setup(context, prefStore, bitShiftingKey)
        } catch (ex: Exception) {
            tryAgain = if (isRecoverableError(ex)) tryRecovery(ex) else throw ex
        }
        if (tryAgain) {
            setup(context, prefStore, bitShiftingKey)
        }
    }
}