package com.flybits.commons.library

import android.content.Context
import android.content.SharedPreferences
import com.flybits.commons.library.utils.Utilities
import com.flybits.internal.db.UserDAO

/**
 * Instances of this class will not encrypt any information and stored sensitive data on the fly
 *
 * @param sharedPreferences [SharedPreferences] to be used for storing keys and values.
 *
 * @param userDAO [UserDAO] for retrieving and storing user data.
 * @param keepSP to keep or clear the saved shared preference
 */
class SharedElementsInMemory internal constructor(
    sharedPreferences: SharedPreferences,
    userDAO: UserDAO,
    private var keepSP: Boolean = true
) : SharedElements(
    sharedPreferences,
    userDAO
) {

    override fun performMigration(context: Context):Int {
        //get all values from backup pref and move to current unsecured pref
        val backupPreferences = context.getSharedPreferences(FLYBITS_STORAGE_UNENCRYPTED_V1, Context.MODE_PRIVATE)
        var migratedCount = 0

        if (backupPreferences.contains(PREF_JWT_TOKEN)) {
            setJWTToken(backupPreferences.getString(PREF_JWT_TOKEN, "") ?: "")
            migratedCount++
        }

        if (backupPreferences.contains(PREF_IDP_CONNECTED)){
            setConnectedIDP(backupPreferences.getString(PREF_IDP_CONNECTED, "") ?: "")
            migratedCount++
        }
        if (backupPreferences.contains(PREF_LANGUAGE_CODES)) {
            setLocalization(Utilities.convertLocalizationStringToList(backupPreferences.getString(PREF_LANGUAGE_CODES, "")))
            migratedCount++
        }
        //no need to migrate gateway URL since it didn't exist yet
        if (migratedCount > 0) {
            backupPreferences.edit().clear().apply()
        }

        if (keepSP) return migratedCount

        if (sharedPreferences.contains(PREF_JWT_TOKEN)){
            sharedPreferences.edit().putString(PREF_JWT_TOKEN,"").apply()
            migratedCount++
        }
        if (sharedPreferences.contains(PREF_IDP_CONNECTED)){
            sharedPreferences.edit().putString(PREF_IDP_CONNECTED,"").apply()
            migratedCount++
        }

        return migratedCount
    }

    override fun setStringVariable(key: String, value: String) {
        when (key) {
            in PREF_JWT_TOKEN, PREF_IDP_CONNECTED -> return
            else -> sharedPreferences.edit().putString(key, value).apply()
        }
    }

    override fun setBooleanVariable(key: String, value: Boolean) {
        when (key) {
            in PREF_JWT_TOKEN, PREF_IDP_CONNECTED -> return
            else -> sharedPreferences.edit().putBoolean(key, value).apply()
        }
    }
}