package com.flybits.commons.library

import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.SharedPreferences
import com.flybits.commons.library.encryption.EncryptionManager
import com.flybits.commons.library.encryption.SecuredPreferenceStore
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.User
import com.flybits.commons.library.utils.Utilities
import com.flybits.internal.db.UserDAO
import com.google.gson.Gson
import kotlinx.coroutines.launch
import java.io.IOException
import java.nio.charset.Charset
import java.security.*
import javax.crypto.BadPaddingException
import javax.crypto.IllegalBlockSizeException
import javax.crypto.NoSuchPaddingException

/**
 * Instances of this class will encrypt all key and values prior to storing them in the [SharedPreferences].
 *
 * @param sharedPreference [SharedPreferences] that will store the data in a secure manner.
 *
 * @param userDAO [UserDAO] for retrieving and storing user data.
 */
class SecuredSharedElements internal constructor(
        private val sharedPreference: SharedPreferences,
        userDAO: UserDAO
) : SharedElements(sharedPreference, userDAO) {

    /**
     * Migrate data from the V1 version of [SecuredSharedElements] to V2, and V0(unencrypted) to V2.
     * Only needed if you used the SDK version less than 1.8.0.
     *
     * @param context Context associated with the application.
     *
     * @param args Map which holds the ARG_PROJECT_ID argument.
     *
     * @return true if the migration succeeded, false otherwise.
     */

    @Throws(IllegalArgumentException::class)

    override fun performMigration(context: Context): Int {
        var migratedValues = 0
        context.getSharedPreferences(FLYBITS_STORAGE_ENC_V2, MODE_PRIVATE).let {
            //perform migration from V2
            if (it.all.isNotEmpty()) {
                job.launch {
                    migratedValues += migrateV3(context)
                }
            }
        }

        return migratedValues
    }

    /**
     * Migrate to version of [SecuredPreferenceStore] that doesn't use project id for key prefix.
     *
     * @param context Context associated with the application.
     *
     * @return number of values migrated.
     */
    internal fun migrateV3(context: Context): Int {
        return try {
            val oldPreferences = context.getSharedPreferences(FLYBITS_STORAGE_ENC_V2, MODE_PRIVATE)

            val encryptionManager = EncryptionManager(context,
                    oldPreferences,
                    FLYBITS_STORAGE_ENC_V2,
                    null,
                    object : SecuredPreferenceStore.KeyStoreRecoveryNotifier {
                        override fun onRecoveryRequired(
                                e: Exception,
                                keyStore: KeyStore?,
                                keyAliases: List<String?>?
                        ): Boolean {
                            return true
                        }
                    })
            var migratedCount = 0

            val jwtHashedKey: String = EncryptionManager.getHashed(PREF_JWT_TOKEN)
            val jwtValue = encryptionManager.decrypt(oldPreferences.getString(jwtHashedKey, ""))
            if (jwtValue != null) {
                SharedElementsFactory.get(context).setJWTToken(jwtValue)
                migratedCount++
            }

            val languageHashedKey: String = EncryptionManager.getHashed(PREF_LANGUAGE_CODES)
            val languageValue = encryptionManager.decrypt(oldPreferences.getString(languageHashedKey, "en"))
            val languages = Utilities.convertLocalizationStringToList(languageValue)
            SharedElementsFactory.get(context).setLocalization(languages)
            migratedCount++

            val userHashedKey: String = EncryptionManager.getHashed(PREF_USER)
            val userValue = encryptionManager.decrypt(oldPreferences.getString(userHashedKey, ""))
            val userObj = Gson().fromJson(userValue, User::class.java)
            SharedElementsFactory.get(context).setUser(userObj)
            migratedCount++

            val locationHashedKey: String = EncryptionManager.getHashed(PREF_LOCATION_PERMISSION)
            val locationValue = encryptionManager.decrypt(oldPreferences.getString(locationHashedKey, ""))
            if (locationValue != null) {
                SharedElementsFactory.get(context).setLocationPermission(locationValue)
                migratedCount++
            }

            val isNotificationHashedKey: String = EncryptionManager.getHashed(PREF_IS_NOTIFICATION_ENABLED)
            val isNotificationValue = encryptionManager.decrypt(oldPreferences.getString(isNotificationHashedKey, "false"))
                    .toBoolean()
            SharedElementsFactory.get(context).setIsPushEnabled(isNotificationValue)
            migratedCount++

            val userIdHashedKey: String = EncryptionManager.getHashed(PREF_USER_ID)
            val userIdValue = encryptionManager.decrypt(oldPreferences.getString(userIdHashedKey, ""))
            if (userIdValue != null) {
                SharedElementsFactory.get(context).setUserId(userIdValue)
                migratedCount++
            }

            val isTokenHashedKey: String = EncryptionManager.getHashed(PREF_IS_TOKEN_SENT)
            val isTokenIdValue = encryptionManager.decrypt(oldPreferences.getString(isTokenHashedKey, ""))
            if (isTokenIdValue != null) {
                SharedElementsFactory.get(context).setPushTokenStatus(isTokenIdValue)
                migratedCount++
            }

            val uniqueIdHashedKey: String = EncryptionManager.getHashed(PREF_UNIQUE_ID)
            val uniqueIdIdValue = encryptionManager.decrypt(oldPreferences.getString(uniqueIdHashedKey, ""))
            if (uniqueIdIdValue != null) {
                SharedElementsFactory.get(context).setUniqueDevice(uniqueIdIdValue)
                migratedCount++
            }

            val gatewayURLHashedKey: String = EncryptionManager.getHashed(PREF_GATEWAY_URL)
            val gatewayURLValue = encryptionManager.decrypt(oldPreferences.getString(gatewayURLHashedKey, ""))
            if (gatewayURLValue != null) {
                SharedElementsFactory.get(context).setGatewayURL(gatewayURLValue)
                migratedCount++
            }

            val idpConnectedHashedKey: String = EncryptionManager.getHashed(PREF_IDP_CONNECTED)
            val idpConnectedValue = encryptionManager.decrypt(oldPreferences.getString(idpConnectedHashedKey, ""))
            if (idpConnectedValue != null) {
                SharedElementsFactory.get(context).setConnectedIDP(idpConnectedValue)
                migratedCount++
            }

            val notificationChannelIdHashedKey: String = EncryptionManager.getHashed(PREF_NOTIFICATION_CHANNEL_ID)
            val notificationChannelIdValue = encryptionManager.decrypt(oldPreferences.getString(notificationChannelIdHashedKey, ""))
            if (notificationChannelIdValue != null) {
                SharedElementsFactory.get(context).setNotificationChannel(notificationChannelIdValue)
                migratedCount++
            }

            // Clear the old shared preference V2.
            oldPreferences.edit().clear().commit()

            migratedCount
        } catch (e: Exception) {
            Logger.e("SecuredSharedElements.migrateV2() error: ${e.message}")
            0
        }
    }

    /**
     *
     * Migrate to version of [SecuredPreferenceStore] that doesn't use project id for key prefix.
     *
     * @param encryptionManager [EncryptionManager] tied to the old project id.
     *
     * @param oldPreferences [SharedPreferences] to be migrated from.
     *
     * @return number of values migrated.
     */

    internal fun migrateV2(
            encryptionManager: EncryptionManager,
            oldPreferences: SharedPreferences
    ): Int {
        data class Key(val hashed: String, val unhashed: String)

        SecuredPreferenceStore.getSharedInstance()?.edit().apply {
            var migratedCount = 0
            setOf(PREF_JWT_TOKEN, PREF_LANGUAGE_CODES, PREF_IDP_CONNECTED).map {
                Key(EncryptionManager.getHashed(it), it)
            }.forEach {
                oldPreferences.getString(it.hashed, null)?.let { encryptedValue ->
                    val decryptedValue = decrypt(encryptedValue, encryptionManager)
                    this?.putString(it.unhashed, decryptedValue)
                    oldPreferences.edit().remove(it.hashed).apply()
                    migratedCount++
                }
            }
            this?.apply()
            Logger.appendTag(TAG).d("migrateV2() migrated $migratedCount values to V2.")
            return migratedCount
        }
    }

    private fun decodeEncryptedText(text: String): EncryptionManager.EncryptedData {
        val parts = text.split("]".toRegex()).dropLastWhile { it.isEmpty() }.toTypedArray()
        var mac: ByteArray? = null
        if (parts.size > 2) {
            mac = EncryptionManager.base64Decode(parts[2])
        }

        return EncryptionManager.EncryptedData(
                EncryptionManager.base64Decode(parts[0]), EncryptionManager.base64Decode(parts[1]), mac
        )
    }

    @Throws(
            IOException::class,
            NoSuchPaddingException::class,
            InvalidKeyException::class,
            NoSuchAlgorithmException::class,
            IllegalBlockSizeException::class,
            BadPaddingException::class,
            EncryptionManager.InvalidMacException::class,
            NoSuchProviderException::class,
            InvalidAlgorithmParameterException::class,
            KeyStoreException::class,
            UnrecoverableEntryException::class
    )
    private fun decrypt(text: String?, encryptionManager: EncryptionManager): String? {
        if (text != null && text.isNotEmpty()) {
            val encryptedData = decodeEncryptedText(text)
            val decrypted = encryptionManager.tryDecrypt(encryptedData)

            return if (decrypted == null) {
                null
            } else String(decrypted, 0, decrypted.size, Charset.defaultCharset())
        }

        return null
    }

    override fun setStringVariable(key: String, value: String) {
        val editor = sharedPreferences.edit()
        editor.putString(key, value)
        editor.apply()
    }

    override fun setBooleanVariable(key: String, value: Boolean) {
        val editor = sharedPreferences.edit()
        editor.putBoolean(key, value)
        editor.apply()
    }
}