package com.flybits.commons.library.utils

import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.os.LocaleList
import com.flybits.commons.library.BuildConfig
import com.flybits.commons.library.SharedElementsFactory.get
import com.flybits.commons.library.analytics.Analytics
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.logging.Logger.exception
import com.flybits.internal.db.CommonsDatabase
import org.json.JSONException
import org.json.JSONObject
import java.io.UnsupportedEncodingException
import java.net.URLDecoder
import java.nio.charset.StandardCharsets
import java.util.Arrays
import java.util.Locale

const val TAG = "UtilityTag"

object Utilities {
    /**
     * Converts a string representation of language codes into a list of language codes.
     *
     * @param languageCodeAsString The string representation of a list of language codes.
     * @return the localization codes as a list, if there are no localization codes set then an
     * empty list will be returned.
     */
    @JvmStatic
    fun convertLocalizationStringToList(languageCodeAsString: String?): ArrayList<String> {
        if (languageCodeAsString == null || languageCodeAsString.isEmpty()) {
            return ArrayList()
        }
        val listOfCodes = languageCodeAsString.split(",").toTypedArray()
        return ArrayList(Arrays.asList(*listOfCodes))
    }

    /**
     * Converts a list of language codes into a single string representation that can be attached
     * to the header of any request.
     *
     * @param listOfCodes The list of codes to convert to String representations.
     * @return the localization codes as String, if there are no localization codes set then an
     * empty string is returned.
     */
    @JvmStatic
    fun convertLocalizationCodeToString(listOfCodes: ArrayList<String>?): String {
        if (listOfCodes != null && listOfCodes.size > 0) {
            var languageCodeString = listOfCodes[0]
            for (i in 1 until listOfCodes.size) {
                languageCodeString = languageCodeString + "," + listOfCodes[i]
            }
            return languageCodeString
        }
        return ""
    }

    /**
     * Get the version of the Flybits Commons SDK
     *
     * @return The string representation of the Flybits Commons SDK Version
     */
    @JvmStatic val sDKVersion: String?
        get() = BuildConfig.FlybitsVersion

    /**
     * Gets the data associated to the device such as the make, model, and OS Version number. This
     * information is sent with every HTTPS request to the user for analytics reasons.
     *
     * @param context The context of the application.
     * @return JSON representation of the User Agent model for all HTTPS requests.
     */
    @JvmStatic
    fun getUserAgentAsJSON(context: Context): String? {
        val obj = JSONObject()
        try {
            obj.put("make", Build.MANUFACTURER)
        } catch (e: JSONException) {
        }
        try {
            obj.put("model", Build.MODEL)
        } catch (e: JSONException) {
        }
        try {
            obj.put("osVersion", Build.VERSION.SDK_INT.toString())
        } catch (e: JSONException) {
        }
        try {
            obj.put("deviceType", "Android")
        } catch (e: JSONException) {
        }
        try {
            val sharedElements = get(context)
            obj.put("physicalDeviceId", sharedElements.getUniqueDeviceId())
        } catch (e: Exception) {
        }
        try {
            obj.put("sdkVersion", sDKVersion)
        } catch (e: JSONException) {
        }
        try {
            val ai = context.packageManager
                    .getApplicationInfo(context.packageName, PackageManager.GET_META_DATA)
            val bundle = ai.metaData
            if (bundle.containsKey("com.flybits.device.name")) {
                val name = bundle.getString("com.flybits.device.name")
                obj.put("name", name)
            } else {
                obj.put("name", Build.MANUFACTURER + "-" + Build.MODEL)
            }
        } catch (e: Exception) {
            exception("getUserAgentAsJSON", e)
        }
        return obj.toString()
    }

    /**
     * Decodes the String if the input String is RichTextFormat Encoded String
     * else returns a plains text.
     *
     * @param rtfString the string passed to be decoded if it in rich text format.
     * @return decoded String or plain text as given as input.
     */
    @JvmStatic
    fun toRTF(rtfString: String): String {
        return try {
            URLDecoder.decode(rtfString, StandardCharsets.UTF_8.name())
        } catch (e: UnsupportedEncodingException) {
            rtfString
        } catch (e: IllegalArgumentException) {
            rtfString
        }
    }

    /**
     * Gets a [ArrayList] of language codes from available [Locale]s on a device.
     *
     * @param context The context of the application.
     * @return languageCodes Array list of languages set in device.
     */
    @JvmStatic
    fun getDeviceLanguageCodes(context: Context): ArrayList<String> {
        val languageCodes = ArrayList<String>()
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            val localeList = LocaleList.getAdjustedDefault()
            for (i in 0 until localeList.size()) {
                val locale = localeList[i]
                languageCodes.add(locale.language)
            }
        } else {
            languageCodes.add(Locale.getDefault().language)
        }
        val sharedElements = get(context)
        sharedElements.setLocalization(languageCodes)
        return languageCodes
    }

    /**
     * Sets json values to the key in the [JSONObject] for request body.
     *
     * @param strKey The [JSONObject] key.
     * @param strValue The [JSONObject] value for the key.
     * @param bodyObject The [JSONObject] to apply key and values to.
     * @return constructed [JSONObject].
     */
    @JvmStatic
    fun setValuesToBody(strKey: String, strValue: String?, bodyObject: JSONObject): JSONObject {
        try {
            bodyObject.put(strKey, strValue)
        } catch (e: JSONException) {
            Logger.appendTag(TAG).d("Invalid JsonObject")
        }
        return bodyObject
    }

    /**
     * Clears or resets [CommonsDatabase] and [SharedElements] values.
     * Called on disconnect from flybits service.
     *
     * @param context Represent Android's context object.
     * @param analytics Represent [Analytics] object as nullable.
     */
    @JvmStatic
    fun clearSDKData(context: Context, analytics: Analytics? = null) {
        analytics?.destroy() ?: run {
            Analytics(context).destroy()
        }
        val sharedElements = get(context)
        val commonsDatabase = CommonsDatabase.getDatabase(context)
        val user = commonsDatabase.userDao().activeUser
        if (user != null) {
            user.isActive = false
            commonsDatabase.userDao().update(user)
        }
        commonsDatabase.ctxDataDAO().deleteAll()
        commonsDatabase.preferenceDAO().clear()
        commonsDatabase.cachingEntryDAO().clear()
        sharedElements.setJWTToken("")
        sharedElements.setConnectedIDP("")
        sharedElements.setUser(null)
    }
}