package com.flybits.commons.library.models.internal;

import androidx.annotation.NonNull;

import com.flybits.commons.library.utils.jbool_expressions.Expression;
import com.flybits.commons.library.utils.jbool_expressions.RuleSet;

import java.util.Set;

public class QueryBuilder<T extends QueryBuilder>{

    public long limit;
    public long offset;

    public String labelsQuery;

    public OrderBy orderBy;
    public SortByEnumeratable sortBy;

    public String cachingKey;
    public int cachingLimit;

    /**
     * Default constructor to initializes all variables.
     */
    public QueryBuilder(){
        limit           = 0;
        offset          = -1;
        cachingLimit    = 10;
    }

    /**
     * Constructor used to initializes all variables using the {@link QueryParameters}.
     *
     * @param parameters The {@link QueryParameters} that are used to construct the
     * {@code QueryBuilder} object.
     */
    public QueryBuilder(QueryParameters parameters){
        limit           = parameters.getLimit();
        offset          = parameters.getOffset();
        orderBy         = parameters.getOrderBy();
        sortBy          = parameters.getSortBy();
        cachingKey      = parameters.getCachingKey();
        cachingLimit    = parameters.getCachingLimit();
        labelsQuery     = parameters.getLabels();
    }

    /**
     * Sets the caching key for the query. This is useful because it allows you to later retrieve
     * cached copies of the query that you just made.
     *
     * @param cachingKey A unique key that represents the query. If no {@code cachingKey} is set a
     *                   hash of the query parameters will be used.
     * @param limit The maximum number of entities that can be cached.
     * @return The {@code QueryBuilder} object that is passed to the {@link QueryParameters} class
     * which is used to construct a GET network query.
     */
    protected T setCaching(@NonNull String cachingKey, int limit){
        this.cachingKey     = cachingKey;
        this.cachingLimit   = limit;
        return (T) this;
    }

    /**
     * Adds a constraint to the results, requiring them to have the given labels.
     *
     * @param label The labels to filter by.
     * @return The {@code QueryBuilder} object that is passed to the {@link QueryParameters} class
     * which is used to construct a GET network query.
     */
    public T setLabels(String label) {

        if (label != null) {
            labelsQuery = "("+label+")";
        }
        return (T) this;
    }

    /**
     * Adds a constraint to the results, requiring them to have the given labels.
     *
     * @param booleanExpression The {@link Expression} query used for retrieving labels.
     * @return The {@code QueryBuilder} object that is passed to the {@link QueryParameters} class
     * which is used to construct a GET network query.
     */
    public T setLabels(Expression booleanExpression) {
        if (booleanExpression != null) {
            Set allVariables = booleanExpression.getAllK();

            if (allVariables.isEmpty()) {
                //Should never happen but just in case
                labelsQuery = null;
            } else if (allVariables.size() == 1) {
                //Prevent bracket from being removed when size of boolean expression is 1. don't use CNF conversion here
                setLabels(allVariables.iterator().next().toString());
            } else {
                //More than one variable, process to CNF
                labelsQuery = RuleSet.toCNF(booleanExpression).toString();
            }
        }
        return (T) this;
    }

    /**
     * Adds a paging mechanism to the request based on the number of returned results wanted and
     * the offset of the next X number of results, where X is a limit.
     *
     * @param limit The maximum number of returned objects.
     * @param offset The offset of where the next X number of response objects will be returned
     *               from where X is the limit.
     * @return The {@code QueryBuilder} object that is passed to the {@link QueryParameters} class
     * which is used to construct a GET network query.
     */
    public T setPaging(long limit, long offset){
        this.limit  = limit;
        this.offset = offset;
        return (T) this;
    }

    /**
     * Sets the server side sorting, so that the data returned is already presorted by some field.
     * @param sort The field to sort by. Each builder will have a SortBy enum.
     * @param order Either OrderBy.Ascending or OrderBy.Descending.
     * @return The {@code QueryBuilder} object that is passed to the {@link QueryParameters} class
     * which is used to construct a GET network query.
     */
    protected T setSorting(@NonNull SortByEnumeratable sort, @NonNull OrderBy order) {
        this.sortBy = sort;
        this.orderBy = order;
        return (T) this;
    }
}