package com.flybits.commons.library.logging

import android.util.Log

/**
 * Class to display information within the Logcat.
 *
 * @param tag The tag under which the log should be displayed in the logcat. The default tag is "FlybitsSDK".
 * @param verbosityLevel The level of verbosity that should be displayed. Log levels enum values from
 * [VerbosityLevel]. The default is level [VerbosityLevel.NONE].
 */
 class Displayer internal constructor(var tag: String = DEFAULT_TAG,
        var verbosity: VerbosityLevel = VerbosityLevel.NONE) {

    companion object {
        const val DEFAULT_TAG = "FlybitsSDK"
    }

    /**
     * Append [filter] to Tag for which the information is logged under within the logcat.
     *
     * @param filter The append filter on top of the `tag` to set for information to be displayed
     * under.
     * @return The [Displayer] class that is used to display information within the logcat.
     */
    fun appendTag(filter: String): Displayer {
        /*
         * There is no mention in official Google Android Log API documentation on the 23 chars tag limit for
         * methods/functions rather than Log.isLoggable for API <= 25 (Nougat, 7.0). Starting from API 26 (Oreo, 8.0.0)
         * this limitation is not presented in the API.
         *
         * However, methods/functions that are intended to sent log messages
         * throw the IllegalArgumentException as well at runtime if the chars limit of the provided tag exceeds 23.
         *
         * Therefore this check is in place to avoid an app unwanted behaviour.
         *
         * https://developer.android.com/reference/android/util/Log
         * https://source.android.com/devices/tech/debug/understanding-logging
         */
        if (filter.length <= 23) {
            this.tag = filter
        } else {
            this.tag = DEFAULT_TAG
        }
        return this
    }

    /**
     * @return The [Displayer] class that is used to display information within the Logcat.
     *
     * Change the verbosity level of the [Displayer], which indicates what should be shown.
     *
     * @param verbosity The level of verbosity that should be displayed. Log levels enum value from [VerbosityLevel]. The
     * default is level [VerbosityLevel.NONE].
     */
    internal fun setVerbosity(verbosity: VerbosityLevel): Displayer {
        this.verbosity = verbosity
        return this
    }

    /**
     * @return True if debug log message with provided [msg] was created and sent to Logcat based on [verbosity]
     * value, false otherwise.
     *
     * @param msg The message to be displayed in the Logcat.
     */
    fun d(msg: String): Boolean {
        return if (isAboveVerbosityLevel(VerbosityLevel.DEBUG)) {
            Log.d(tag, msg)
            true
        } else {
            false
        }
    }

    /**
     * @return True if error log message with provided [msg] was created and sent to Logcat based on [verbosity]
     * value, false otherwise.
     *
     * @param msg The message to be displayed in the Logcat.
     */
    fun e(msg: String): Boolean {
        return if (isAboveVerbosityLevel(VerbosityLevel.ERROR)) {
            Log.e(tag, msg)
            true
        } else {
            false
        }
    }

    /**
     * @return True if error log message with provided [msg] and [exception] was created and sent to Logcat based on
     * [verbosity] value, false otherwise.
     *
     * @param msg The message to be displayed in the logcat.
     * @param exception An exception that is associated to the error
     */
    fun e(msg: String, exception: Exception): Boolean {
        return e("$msg - ${exception.localizedMessage}")
    }

    /**
     * @return True if information log message with provided [msg] was created and sent to Logcat based on [verbosity]
     * value, false otherwise.
     *
     * @param msg The message to be displayed in the logcat.
     */
    fun i(msg: String): Boolean {
        return if (isAboveVerbosityLevel(VerbosityLevel.INFO)) {
            Log.i(tag, msg)
            true
        } else {
            false
        }
    }

    /**
     * @return True if warning log message with provided [msg] was created and sent to Logcat based on [verbosity]
     * value, false otherwise.
     *
     * @param msg The message to be displayed in the logcat.
     */
    fun w(msg: String): Boolean {
        return if (isAboveVerbosityLevel(VerbosityLevel.WARNINGS)) {
            Log.w(tag, msg)
            true
        } else {
            false
        }
    }

    /**
     * Creates a log statement in the logcat that displays an exception including the
     * stacktrace.
     *
     * @param sdkMethod The class/method identifier that indicates where the exception
     * occurred.
     * @param e The exception that was thrown
     * @return Whether or not the message was displayed based [verbosity] value.
     */
    fun exception(sdkMethod: String, e: Exception): Boolean {
        return if (isAboveVerbosityLevel(VerbosityLevel.FATAL)) {
            Log.e(tag, "There was an exception thrown for: $sdkMethod")
            e.printStackTrace()
            true
        } else {
            false
        }
    }

    private fun isAboveVerbosityLevel(checkVerbosity: VerbosityLevel): Boolean {
        return verbosity.level >= checkVerbosity.level
    }
}
