package com.flybits.commons.library.api

import android.content.Context
import android.util.Log
import android.webkit.URLUtil
import com.flybits.commons.library.SharedElementsFactory.get
import com.flybits.commons.library.models.PushProviderType
import java.util.Locale

/**
 *  A class to hold the configuration for Flybits project, initialized either by including the 'FlybitsConfiguration.json' file in assets folder, or through the builder
 *  - Important: Values set through builder pattern will override values found in 'FlybitsConfiguration.json' file.
 */
class FlybitsConfiguration private constructor(builder: Builder) {

    companion object {
        const val CONFIGURATION_FILE_NAME = "FlybitsConfiguration.json"

        const val CONFIGURATION_PARSER_TAG = "FlybitsConfigParser"

        const val PROJECT_CONFIG_JSON = "projectConfiguration"
        const val PROJECT_ID_JSON = "projectId"
        const val GATEWAY_URL_JSON = "gatewayUrl"
        const val PUSH_PROVIDER = "pushProvider"
    }

    val applicationContext: Context = builder.applicationContext
    var projectId: String = Builder.projectId
    var gatewayUrl: String = Builder.gatewayUrl
    var pushProviderFromJson = Builder.pushProviderFromJson

    /**
     * Builder class for [FlybitsConfiguration]
     *
     * @param applicationContext context of host application
     */
    class Builder(val applicationContext: Context) {

        /**
         *  Sets the projectId
         *
         *  @param projectId The project id string
         */
        fun setProjectId(projectId: String): Builder {
            Builder.projectId = projectId
            return this
        }

        /**
         *  Sets the gatewayUrl
         *
         *  @param gatewayUrl The gateway url string
         */
        fun setGateWayUrl(gatewayUrl: String): Builder {
            if (!URLUtil.isHttpsUrl(gatewayUrl)) {
                val errorMessage = """
                   🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨
                   🚨 Using a non HTTPS url: $gatewayUrl
                   🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨🚨
                """.trimIndent()
                Log.w("FlybitsConfiguration", errorMessage)
            }
            Builder.gatewayUrl = gatewayUrl
            return this
        }

        /**
         *  Sets the pushProviderType
         *
         *  @param pushProviderType The Push Provider type specified.
         */
        fun setPushProviderType(pushProviderType: String): Builder {
            when (pushProviderType.lowercase(Locale.getDefault())) {
                "gcm" -> get(applicationContext).storePushProviderType(PushProviderType.GCM)
                "fcm" -> get(applicationContext).storePushProviderType(PushProviderType.FCM)
                "huawei" -> get(applicationContext).storePushProviderType(PushProviderType.HUAWEI)
            }
            return this
        }

        fun build(): FlybitsConfiguration {
            return FlybitsConfiguration(this)
        }

        companion object {
            var projectId: String = ""
                private set

            var gatewayUrl: String = ""
                private set

            var pushProviderFromJson: String = ""
                internal set
        }
    }
}