package com.flybits.commons.library.analytics

import android.content.Context
import androidx.work.*
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.CtxData
import com.flybits.commons.library.models.State
import com.flybits.internal.db.CommonsDatabase
import kotlinx.coroutines.CoroutineExceptionHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import org.json.JSONObject
import java.util.concurrent.TimeUnit

const val TAG_ANALYTICS = "Analytics"
const val UPLOAD_INTERVAL_METERED = 60L
const val UPLOAD_INTERVAL_UNMETERED = 20L
const val CTX_DATA_ENDPOINT = "/context/ctxdata"

/**
 * This class is responsible for tracking analytics data, this including: scheduling workers
 * , storing the data in the local database, and formatting the data in the expected format.
 */
open class Analytics(private val context: Context) {

    private fun exceptionHandlerBuilder(text: String = "Error executing the CoroutineScope") =
            CoroutineExceptionHandler { _, _ ->
                Logger.appendTag(TAG_ANALYTICS).e(text)
            }

    private val ctxDataDAO = CommonsDatabase.getDatabase(context).ctxDataDAO()

    companion object {
        const val TAG_METERED_WORK = "com.flybits.commons.library.analytics.meteredWork"
        const val TAG_UNMETERED_WORK = "com.flybits.commons.library.analytics.unmeteredWork"
    }

    /**
     * Represents an application event that will be tracked by the Flybits Analytics system.
     *
     * @param analyticsScope Scope of the analytics e.x push or content.
     * @param action the key of JSON object going to send
     * @param value the value of the action
     * @param timestamp When the event occurred in epoch time in milliseconds.
     *
     */

    data class AnalyticsEvent(
            val analyticsScope: String,
            val action: String,
            val value: Boolean = true,
            val timestamp: Long,
            val valueString: String = ""
    )

    private fun AnalyticsEvent.toCtxData(): CtxData {
        val analyticsJson = JSONObject()
        val dataTypeID = "ctx.flybits.${this.analyticsScope}"
        analyticsJson.put("dataTypeID", dataTypeID)
        val valueJson = JSONObject()
        if (valueString.isNotEmpty()) {
            valueJson.put(action, valueString)
        } else {
            valueJson.put(action, value)
        }
        analyticsJson.put("value", valueJson)
        analyticsJson.put(
                "timestamp",
                timestamp / 1000
        ) // Convert the timestamp to seconds prior sending to the server as required.
        return CtxData(timestamp = timestamp, value = analyticsJson.toString(), pluginID = dataTypeID)
    }

    /**
     * Flush all data from the ctxDatabase to the server then delete the data if it was
     * sent with success.
     *
     * @param callback Callback which will be informed about whether the flush succeeded or failed.
     * @param async Whether or not to execute on a separate thread, true by default.
     *
     */
    @Throws(FlybitsException::class)
    fun flush(
            callback: ((success: Boolean, exception: Exception?) -> Unit)? = null,
            async: Boolean = true
    ) {
        Logger.appendTag(TAG_ANALYTICS).i("Flushing analytics data")
        if (async) CoroutineScope(Dispatchers.IO).launch(exceptionHandlerBuilder("Failed to execute CoroutineScope in Analytics.flush().")) {
            try {
                flush(callback)
            } catch (e: Exception) {
                callback?.invoke(false, e)
                Logger.appendTag(TAG_ANALYTICS).e("Flushing analytics data failed", e)
            }
        } else flush(callback)
    }

    @Throws(FlybitsException::class)
    private fun flush(callback: ((success: Boolean, exception: Exception?) -> Unit)? = null) {
        try {
            val ctxDataDAO = CommonsDatabase.getDatabase(context).ctxDataDAO()
            val ctxData = ctxDataDAO.getAllByState(State.PENDING)

            if (!ctxData.isNullOrEmpty()) {
                Logger.appendTag(TAG_ANALYTICS).i("Flushing analytics data - Non-Empty list")
                ctxData.forEach {
                    it.state = State.INFLIGHT
                }
                ctxDataDAO.updateCtxData(ctxData)

                val body = ctxData.joinToString(separator = ",", prefix = "[", postfix = "]") { it.value }
                val result = FlyAway.post<Any>(
                        context,
                        CTX_DATA_ENDPOINT,
                        body,
                        null,
                        "AnalyticsWorker.doWork",
                        null
                )

                if (result.status == RequestStatus.COMPLETED) {
                    //We cannot just delete all since by the time the response comes back an item which has not been sent may have been added
                    ctxDataDAO?.deleteMany(ctxData)
                    callback?.invoke(true, null)
                    Logger.appendTag(TAG_ANALYTICS).i("Flushing analytics data success")
                } else {
                    ctxData.forEach {
                        it.state = State.PENDING
                    }
                    ctxDataDAO.updateCtxData(ctxData)

                    callback?.invoke(false, result.exception)
                    Logger.appendTag(TAG_ANALYTICS)
                            .e("Flushing analytics data failed", result.exception)
                }
            } else {
                callback?.invoke(true, null)
                Logger.appendTag(TAG_ANALYTICS)
                        .i("Flushing analytics data - Empty list, nothing sent")
            }
        } catch (e: Exception) {
            callback?.invoke(false, e)
            Logger.appendTag(TAG_ANALYTICS).e("Flushing analytics data failed", e)
        }
    }

    /**
     * Enqueue a {@link PeriodicWorkRequest} which involves a worker periodically
     * uploading data from the local database to the server pertaining to analytics.
     * Data will be sent to the server for both a metered and unmetered connection.
     * Metered connection data will be sent less frequently.
     *
     * @return false if both metered and unmetered workers have been scheduled, true otherwise.
     */
    fun scheduleWorkers() {
        scheduleMeteredWorker()
        scheduleUnmeteredWorker()
        Logger.appendTag(TAG_ANALYTICS).i("Analytics workers are scheduled")
    }

    fun cancelWorkers() {
        cancelUnmeteredWork()
        cancelMeteredWork()
        Logger.appendTag(TAG_ANALYTICS).i("Analytics workers are cancelled")
    }

    /**
     * Destroy all stored analytics data, and workers responsible for uploading it.
     *
     */
    fun destroy() {
        cancelWorkers()
        CoroutineScope(Dispatchers.IO).launch(exceptionHandlerBuilder("Failed to execute CoroutineScope in Analytics.destroy().")) {
            ctxDataDAO?.deleteAll()
            Logger.appendTag(TAG_ANALYTICS).i("Ctxdata database in Commons cleared")
        }
    }

    private fun cancelUnmeteredWork() {
        WorkManager.getInstance(context).cancelUniqueWork(TAG_UNMETERED_WORK)
    }

    private fun cancelMeteredWork() {
        WorkManager.getInstance(context).cancelUniqueWork(TAG_METERED_WORK)
    }

    private fun scheduleMeteredWorker() {
        val constraintsMetered = Constraints.Builder()
                .setRequiredNetworkType(NetworkType.METERED)
                .build()

        val workRequestMetered = PeriodicWorkRequest
                .Builder(ContextReporterWorker::class.java, UPLOAD_INTERVAL_METERED, TimeUnit.MINUTES)
                .setConstraints(constraintsMetered)
                .addTag(TAG_METERED_WORK)
                .build()

        WorkManager.getInstance(context).enqueueUniquePeriodicWork(
                TAG_METERED_WORK,
                ExistingPeriodicWorkPolicy.KEEP,
                workRequestMetered
        )
    }

    private fun scheduleUnmeteredWorker() {
        val constraintsMetered = Constraints.Builder()
                .setRequiredNetworkType(NetworkType.UNMETERED)
                .build()

        val workRequestMetered = PeriodicWorkRequest
                .Builder(ContextReporterWorker::class.java, UPLOAD_INTERVAL_UNMETERED, TimeUnit.MINUTES)
                .setConstraints(constraintsMetered)
                .addTag(TAG_UNMETERED_WORK)
                .build()

        WorkManager.getInstance(context).enqueueUniquePeriodicWork(
                TAG_UNMETERED_WORK,
                ExistingPeriodicWorkPolicy.KEEP,
                workRequestMetered
        )
    }

    /**
     * Queue an event for tracking which will be uploaded to the server in the future.
     *
     * Runs on a separate thread.
     *
     * @param event Analytics event that you want to track.
     */
    protected fun track(event: AnalyticsEvent) {
        CoroutineScope(Dispatchers.IO).launch(exceptionHandlerBuilder("Failed to execute CoroutineScope in Analytics.track().")) {
            try {
                val data = event.toCtxData()
                Logger.appendTag(TAG_ANALYTICS).i("Received event: $data")
                if (checkIfUnique(data)) {
                    ctxDataDAO?.insert(data)
                    Logger.appendTag(TAG_ANALYTICS).i("Recorded event: ${data.id} - ${data.value}")
                }
            } catch (e: Exception) {
                Logger.appendTag(TAG_ANALYTICS).e("Tracking analytics data failed", e)
            }
        }
    }

    internal fun checkIfUnique(ctxData: CtxData): Boolean {
        synchronized(this) {
            val dataFromCursor = ctxDataDAO?.getDataByValue(ctxData.value)
            return dataFromCursor == null
        }
    }
}