package com.flybits.commons.library.api;

import com.flybits.commons.CommonScope;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.SharedElementsFactory;
import com.flybits.commons.library.UnsecuredSharedElements;
import com.flybits.commons.library.analytics.Analytics;
import com.flybits.commons.library.api.idps.HttpMethod;
import com.flybits.commons.library.api.idps.IDP;
import com.flybits.commons.library.api.idps.JwtIDP;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.deserializations.DeserializeLogin;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.exceptions.InvalidFlybitsManagerException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.logging.VerbosityLevel;
import com.flybits.commons.library.models.Jwt;
import com.flybits.commons.library.models.PushProviderType;
import com.flybits.commons.library.models.User;
import com.flybits.commons.library.models.UserOptedIn;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.commons.library.utils.Utilities;
import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.CtxDataDAO;
import com.flybits.internal.db.UserDAO;

import org.json.JSONException;
import org.json.JSONObject;

import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.os.LocaleList;
import android.os.Looper;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import static com.flybits.commons.library.SharedElements.PREF_IDP_CONNECTED;
import static com.flybits.commons.library.SharedElements.PREF_JWT_TOKEN;

/**
 * The {@code FlybitsManager} class is responsible for defining all {@link FlybitsScope}s as well as
 * performing all {@link User} operations such as logging in, logging out, disabling user's account,
 * user forgotten password, and refreshing the logged in user's JWT.
 */
public class FlybitsManager {
    public static final String AUTHENTICATION_API = "/sso/auth";
    static final String DISCONNECT_ENDPOINT = AUTHENTICATION_API + "/logout";
    static final String DISABLE_PUSH = "/push/token";
    private static final String TAG_MANAGER = "Manager";
    private static final String ACTIVE_API = AUTHENTICATION_API + "/activationStatus";

    private Context context;
    private IDP idProvider;
    private Analytics analytics;
    private ArrayList<String> languageCodes;
    private SharedElements sharedElements;
    private boolean fallBackSPEncrypToMemo = false;
    private boolean keepSP = true;

    /**
     * Constructor used to define the {@code FlybitsManager} based on the {@link Builder}
     * attributes.
     *
     * @param builder The {@link Builder} that defines all the attributes about the
     *                {@code FlybitsManager}.
     */
    private FlybitsManager(Builder builder) {
        context = builder.mContext;
        languageCodes = builder.languageCodes;
        idProvider = builder.idProvider;
        analytics = new Analytics(context);
        fallBackSPEncrypToMemo = builder.failedEncryptedToMemory;
        keepSP = builder.retainSharedPref;
        sharedElements = SharedElementsFactory.INSTANCE.get(context, fallBackSPEncrypToMemo, keepSP);

        if (languageCodes.size() == 0) {
            languageCodes.add("en");
        }
        sharedElements.migrateData(context);
        sharedElements.setLocalization(languageCodes);
    }

    private static void parseConfigJson(FlybitsConfiguration flybitsConfiguration, Context context) {
        String configJson = null;
        try {
            InputStream is = flybitsConfiguration.getApplicationContext().getAssets()
                    .open(FlybitsConfiguration.CONFIGURATION_FILE_NAME);
            BufferedReader reader = new BufferedReader(new InputStreamReader(is));
            StringBuilder stringBuilder = new StringBuilder();
            String line;
            String ls = System.getProperty("line.separator");
            while ((line = reader.readLine()) != null) {
                stringBuilder.append(line);
                stringBuilder.append(ls);
            }

            stringBuilder.deleteCharAt(stringBuilder.length() - 1);
            reader.close();
            configJson = stringBuilder.toString();
        } catch (IOException e) {
            Logger.appendTag(FlybitsConfiguration.CONFIGURATION_PARSER_TAG)
                    .e("No " + FlybitsConfiguration.CONFIGURATION_FILE_NAME +
                            " found, values can be set using builder pattern");
        }

        if (configJson != null) {
            try {
                JSONObject jsonObject = new JSONObject(configJson)
                        .getJSONObject(FlybitsConfiguration.PROJECT_CONFIG_JSON);
                if (flybitsConfiguration.getProjectId().isEmpty()) {
                    flybitsConfiguration.setProjectId(jsonObject.getString(FlybitsConfiguration.PROJECT_ID_JSON));
                }
                if (flybitsConfiguration.getGatewayUrl().isEmpty()) {
                    flybitsConfiguration.setGatewayUrl(jsonObject.getString(FlybitsConfiguration.GATEWAY_URL_JSON));
                }
                if (flybitsConfiguration.getPushProviderFromJson().isEmpty()) {
                    try {
                        flybitsConfiguration.setPushProviderFromJson(jsonObject.getString(FlybitsConfiguration.PUSH_PROVIDER));
                    } catch (Exception e) {
                        flybitsConfiguration.setPushProviderFromJson("FCM");
                        Logger.appendTag(FlybitsConfiguration.CONFIGURATION_PARSER_TAG)
                                .e("No ${FlybitsConciergeConfiguration.PUSH_PROVIDER} found.");
                    }
                    switch (flybitsConfiguration.getPushProviderFromJson().toLowerCase()) {
                        case "gcm":
                            SharedElementsFactory.INSTANCE.get(context).storePushProviderType(PushProviderType.GCM);
                            break;
                        case "fcm":
                            SharedElementsFactory.INSTANCE.get(context).storePushProviderType(PushProviderType.FCM);
                            break;
                        case "huawei":
                            SharedElementsFactory.INSTANCE.get(context)
                                    .storePushProviderType(PushProviderType.HUAWEI);
                            break;
                    }
                }
            } catch (JSONException | IllegalArgumentException e) {
                Logger.appendTag(FlybitsConfiguration.CONFIGURATION_PARSER_TAG)
                        .e("Failed to parse FlybitsConfiguration.", e);
            }
        }
    }

    /**
     * Configure Flybits by supplying a {@link FlybitsConfiguration} object
     *
     * @param flybitsConfiguration {@link FlybitsConfiguration} object that contains necessary variables for configuring Flybits
     */
    public static void configure(@Nullable FlybitsConfiguration flybitsConfiguration, Context context) {
        FlybitsConfiguration config;
        if (flybitsConfiguration == null) {
            config = new FlybitsConfiguration.Builder(context).build();
        } else {
            config = flybitsConfiguration;
        }
        parseConfigJson(config, context);

        SharedElements sharedElements = SharedElementsFactory.INSTANCE.get(context);

        if (!config.getGatewayUrl().equals("")) {
            if (!sharedElements.getGatewayURL().equals(config.getGatewayUrl()) &&
                    !sharedElements.getSavedJWTToken().isEmpty()) {
                Logger.appendTag(TAG_MANAGER)
                        .e("You are trying to change the gatewayUrl with a user already connected to Flybits" +
                                "gatewayUrl:" + sharedElements.getGatewayURL() +
                                "You must disconnect your current session " +
                                "before connecting to a new gatewayUrl");
                return;
            }
        }
        sharedElements.setGatewayURL(config.getGatewayUrl());

        // If the user is active and there is no JWT,
        // there should be no valid session and the user should be set to inactive
        if (sharedElements.getSavedJWTToken().isEmpty()) {
            CountDownLatch latch = new CountDownLatch(1);

            final ExecutorService executorService = Executors.newSingleThreadExecutor();
            executorService.execute(() -> {
                CommonsDatabase commonsDatabase = CommonsDatabase.getDatabase(context);
                User user = commonsDatabase.userDao().getActiveUser();
                if (user != null) {
                    commonsDatabase.userDao().resetAllUsers();
                }
                latch.countDown();
            });
            try {
                latch.await();
            } catch (InterruptedException e) {
                Logger.exception("FlybitsManager.configure", e);
            }
        }
    }

    /**
     * Add a {@link FlybitsScope} to the {@link FlybitsManager}. The {@link FlybitsScope}
     * defines the SDK and all the properties associated to it.
     *
     * @param scope The {@link FlybitsScope} that is associated to this application.
     */
    public static void addScope(@NonNull FlybitsScope scope, Context context) {
        ScopeOfficer.addScope(scope, context);
    }

    /**
     * Remove a {@link FlybitsScope} to the {@link FlybitsManager}. The {@link FlybitsScope}
     * defines the SDK and all the properties associated to it.
     *
     * @param scope The {@link FlybitsScope} that is associated to this application.
     */
    public static void removeScope(@NonNull FlybitsScope scope) {
        ScopeOfficer.removeScope(scope);
    }

    /**
     * Set the logging verbosity for all messages within the SDKs.
     *
     * @param level The level of verbosity that should be displayed. Log levels enum values
     *              from [VerbosityLevel]. The default is level [VerbosityLevel.NONE].
     */
    public static void setLoggingVerbosity(VerbosityLevel level) {
        Logger.setVerbosity(level);
    }

    /**
     * Get size of {@link FlybitsScope}.
     *
     * @return size of scope The {@link FlybitsScope} that is associated to this application.
     */
    static int getScopeSize() {
        return ScopeOfficer.getFlybitsScopes().size();
    }

    /**
     * Get if {@link FlybitsScope} is added by the {@link ScopeOfficer}.
     * This is to determine if the ConciergeScope is added for Opted In/Out Process.
     *
     * @param scopeName It is associated with {@link FlybitsScope}.
     *
     * @return true if scope exists in FlybitsScope referred by scope name.
     */
    public static boolean isScopeExists(String scopeName) {
        return ScopeOfficer.getFlybitsScopes().containsKey(scopeName);
    }

    /**
     * @param context The Context of the application.
     *
     * @return true if there is a valid {@link Jwt} (an active connection to Flybits' server), otherwise false.
     */
    public static boolean isConnected(@NonNull final Context context) {
        try {
            final String savedJwt = SharedElementsFactory.INSTANCE.get(context).getSavedJWTToken();

            if (!savedJwt.isEmpty()) { // Make sure there is a JWT locally.
                // Only return true if the JWT decoded is not expired.
                return Jwt.decodeJWTToken(savedJwt).getExpiry() > new Date().getTime() / 1000;
            }
            return false;
        } catch (final Exception e) {
            Logger.appendTag(TAG_MANAGER).e("Failed to retrieve JWT information: " + e.toString());
            return false;
        }
    }

    /**
     * Clear the sensitive data previously saved in the Shared preference
     * This should only be called after a {@link SharedElements]} object get initialized
     * i.e. after {@link #isConnected} get called
     */
    public static void clearUnsecuredSharePref() {
        SharedElements sharedElements = SharedElementsFactory.savedInstance;
        if (sharedElements != null && sharedElements instanceof UnsecuredSharedElements) {
            sharedElements.setStringVariable(PREF_JWT_TOKEN, "");
            sharedElements.setStringVariable(PREF_IDP_CONNECTED, "");
        }
    }

    /**
     * Opt the currently connected user out of Flybits. All local data associated to the user
     * will be cleared. All communications with Flybits will be stopped until
     * {@link FlybitsManager#optIn(BasicResultCallback, Handler)} is called.
     *
     * @param callback The callback used to indicate whether or not the opt out was
     *                 successful or not.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult optOut(final BasicResultCallback callback) {
        return optOut(callback, new Handler());
    }

    /**
     * Opt the currently connected user out of Flybits. All local data associated to the user
     * will be cleared. All communications with Flybits will be stopped until
     * {@link FlybitsManager#optIn(BasicResultCallback, Handler)} is called.
     *
     * @param callback The callback used to indicate whether or not the opt out was
     *                 successful or not.
     * @param handler  The handler that the callback result will be posted on.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult optOut(final BasicResultCallback callback, final Handler handler) {
        return setOptedState(false, callback, handler);
    }

    /**
     * Opt in the currently connected user back into Flybits. All communications with Flybits
     * servers will be restored.
     *
     * @param callback The callback used to indicate whether or not the opt in was
     *                 successful or not.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult optIn(final BasicResultCallback callback) {
        return optIn(callback, new Handler());
    }

    /**
     * Opt in the currently connected user back into Flybits. All communications with Flybits
     * servers will be restored.
     *
     * @param callback The callback used to indicate whether or not the opt in was
     *                 successful or not.
     * @param handler  The handler that the callback result will be posted on.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult optIn(final BasicResultCallback callback, Handler handler) {
        return setOptedState(true, callback, handler);
    }

    /**
     * Set the opted state of the connected user out. Opting out(optedState=false) will result in
     * all local data associated to the user being cleared. All communications with Flybits will
     * be stopped until the state is returned to back to opted in. .
     * {@link FlybitsManager#optIn(BasicResultCallback, Handler)} is called.
     *
     * @param optedState True to opt in, false to opt out.
     * @param callback   The callback used to indicate whether or not the opt out was
     *                   successful or not.
     * @param handler    Handler that the callback result will be posted on.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult setOptedState(final boolean optedState, final BasicResultCallback callback, final Handler handler) {
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult query = new BasicResult(callback, handler, executorService);
        executorService.execute(() -> {
            Result result = FlyAway.post(context, ACTIVE_API
                    , String.format("{\"%s\":%s}", UserOptedIn.FIELD_IS_ACTIVE, optedState), null
                    , "FlybitsManager.setOptedState", null);
            if (result.getStatus() == RequestStatus.COMPLETED) {

                CommonsDatabase commonsDatabase = CommonsDatabase.getDatabase(context);
                if (!optedState) {
                    //Clear database data & destroy analytics
                    analytics.destroy();
                    commonsDatabase.ctxDataDAO().deleteAll();
                    commonsDatabase.preferenceDAO().clear();
                    commonsDatabase.cachingEntryDAO().clear();
                } else {
                    analytics.scheduleWorkers();
                }

                //Update user model in DB so that the isOptedIn flag is set to optedState passed
                User user = commonsDatabase.userDao().getActiveUser();
                if (user != null) {
                    user.setOptedIn(optedState);
                    commonsDatabase.userDao().update(user);
                }

                //Update user model in SharePreference so that the isOptedIn flag is set to optedState passed
                SharedElements sharedElements = SharedElementsFactory.INSTANCE.get(context);
                User userObj = sharedElements.getUser();
                if (userObj != null) {
                    userObj.setOptedIn(optedState);
                    sharedElements.setUser(userObj);
                }

                ScopeOfficer.INSTANCE.onOptedStateChange(context, optedState);
                query.setResult(result);
            } else {
                query.setFailed(result.getException());
            }
        });
        return query;
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param callback The response callback to run after a connection is made or fails.
     * @param handler  The handler that the callback will be invoked through.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult connect(final BasicResultCallback callback, @NonNull final Handler handler) {
        return connect(callback, true, handler, null);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param callback The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult connect(final BasicResultCallback callback) {
        return connect(callback, true);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param idp      The {@link IDP} that should be used for registering contextual information to the
     *                 account.
     * @param callback The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult connect(@NonNull final IDP idp, final BasicResultCallback callback) {
        idProvider = idp;
        return connect(callback, true);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param idp        The {@link IDP} that should be used for registering contextual information to the
     *                   account.
     * @param customerId The unique identifier assigned by the client that represents user and it does not represent flybits user id.
     * @param callback   The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult connect(@NonNull final IDP idp, @NonNull String customerId, final BasicResultCallback callback) {
        return connect(idp, callback, customerId, new Handler(Looper.getMainLooper()));
    }

    BasicResult connect(@NonNull final IDP idp, final BasicResultCallback callback, @NonNull String customerId,
            @NonNull final Handler handler) {
        idProvider = idp;
        return connect(callback, true, handler, customerId);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method.
     *
     * @param idp        The {@link IDP} that should be used for registering contextual information to the account.
     * @param customerId The unique identifier assigned by the client that represents user and it does not represent
     *                   flybits user id.
     * @param deviceId   The {@link String} representation of device ID to be persisted in {@link SharedElements#setUniqueDevice(String)}.
     * @param callback   The response callback to run after a connection is made or fails.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult connect(@NonNull final IDP idp, @Nullable final String customerId,
            @NonNull final String deviceId, @NonNull final BasicResultCallback callback) {
        idProvider = idp;
        sharedElements.setUniqueDevice(deviceId);
        return connect(callback, true, new Handler(Looper.getMainLooper()), customerId);
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method. If autoUseManifestProject is true,
     * the project to load will be pulled from the manifest. Otherwise it will be in a no-project
     * state. This can only be done with the Flybits IDP.
     *
     * @param callback               The response callback to run after a connection is made or fails.
     * @param autoUseManifestProject If true, the project id defined in the manifest will be loaded
     *                               automatically.
     * @param handler                The handler that the callback will be invoked through.
     * @param customerId             The unique identifier assigned by the client that represents user and it does not represent flybits user id.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult connect(final BasicResultCallback callback, final boolean autoUseManifestProject,
            @NonNull final Handler handler, String customerId) {

        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult query = new BasicResult(callback, handler, executorService);
        executorService.execute(() -> {
            if (isConnected(context)) {
                FlybitsException e = new FlybitsException("You already have an IDP connected to " +
                        "the Flybits SDK, please call the disconnect method before calling connect again");
                query.setFailed(e);
            } else {
                if (idProvider != null) {
                    ScopeOfficer.INSTANCE.onConnecting();
                    String body = "";
                    if (idProvider.getBody() != null) {
                        body = idProvider.getBody().toString();
                    }
                    String url = AUTHENTICATION_API + idProvider.getAuthenticationEndPoint();
                    final Result<User> authenticatedUser;
                    HttpMethod requestType = idProvider.getRequestType();
                    if (idProvider instanceof JwtIDP) {
                        url = idProvider.getAuthenticationEndPoint();
                        Result updateUserAgent =
                                FlyAway.post(context, url, "", null,
                                        "Device.userAgent", null,
                                        idProvider.getHeaders());
                        if (updateUserAgent.getStatus() == RequestStatus.COMPLETED) {
                            String userEndpoint =
                                    AUTHENTICATION_API + ((JwtIDP) idProvider).getUserInfoEndPoint();
                            authenticatedUser = FlyAway.get(context, userEndpoint,
                                    idProvider.getHeaders(), new DeserializeLogin(),
                                    "FlybitsManager.connect", User.class);
                            connectUserAuthenticate(authenticatedUser, customerId, query);
                        } else {
                            sharedElements.setJWTToken("");
                            sharedElements.setUser(null);
                            ScopeOfficer.INSTANCE.onFailedToConnect(context);
                            query.setFailed(new FlybitsException("Authentication Failed."));
                        }
                    } else {
                        if (requestType.equals(HttpMethod.GET)) {
                            authenticatedUser = FlyAway.get(context, url,
                                    idProvider.getHeaders(), new DeserializeLogin(),
                                    "FlybitsManager.connect", User.class);
                        } else if (requestType.equals(HttpMethod.PUT)) {
                            authenticatedUser = FlyAway.put(context, url, body,
                                    new DeserializeLogin(),
                                    "FlybitsManager.connect", User.class, idProvider.getHeaders());
                        } else {
                            authenticatedUser = FlyAway.post(context, url, body,
                                    new DeserializeLogin(),
                                    "FlybitsManager.connect", User.class, idProvider.getHeaders());
                        }
                        connectUserAuthenticate(authenticatedUser, customerId, query);
                    }
                }
            }
        });
        return query;
    }

    private void connectUserAuthenticate(Result<User> authenticatedUser, String customerId, BasicResult query) {
        if (authenticatedUser.getStatus() == RequestStatus.COMPLETED) {
            User user = authenticatedUser.getResponse();
            UserDAO userDAO = CommonsDatabase.getDatabase(context).userDao();

            userDAO.resetAllUsers();

            CtxDataDAO ctxDataDAO = CommonsDatabase.getDatabase(context).ctxDataDAO();
            ctxDataDAO.deleteAll();

            if (customerId != null && !customerId.isEmpty()) {
                user.setCustomerID(customerId);
            }
            userDAO.insert(user);

            sharedElements.setConnectedIDP(idProvider.getProvider());
            //Setting active user to shared preference.
            sharedElements.setUser(user);
            sharedElements.setUserId(authenticatedUser.getResponse().getId());

            idProvider.onAuthenticated(context, user);

            analytics.scheduleWorkers();

            ScopeOfficer.INSTANCE.onConnected(context, user);
            query.setResult(authenticatedUser);
        } else {
            ScopeOfficer.INSTANCE.onFailedToConnect(context);
            query.setFailed(authenticatedUser.getException());
        }
    }

    /**
     * The {@code connect} method is responsible for connecting the application, implemented with
     * this SDK, to the Flybits system. Once a successful connection has been made, all registered
     * {@link FlybitsScope}s will be notified through the
     * {@link FlybitsScope#onConnected(Context, User)} method. If autoUseManifestProject is true,
     * the project to load will be pulled from the manifest. Otherwise it will be in a no-project
     * state. This can only be done with the Flybits IDP.
     *
     * @param callback               The response callback to run after a connection is made or fails.
     * @param autoUseManifestProject If true, the project id defined in the manifest will be loaded
     *                               automatically.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult connect(final BasicResultCallback callback, final boolean autoUseManifestProject) {

        if (idProvider == null) {
            final ExecutorService executorService = Executors.newSingleThreadExecutor();
            final BasicResult query = new BasicResult(callback, new Handler(Looper.getMainLooper()), executorService);

            FlybitsException e = new FlybitsException("You must enter an IDP in the Flybits Builder to use this connect() method.");
            callback.onException(e);

            return query;
        }

        return connect(callback, autoUseManifestProject, new Handler(Looper.getMainLooper()), null);
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context)}
     * method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     *                 successful or not.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback) {
        return disconnect(callback, false);
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context)} method.
     *
     * @param callback              The callback used to indicate whether or not the disconnection was
     *                              successful or not.
     * @param disconnectOnException true if the disconnect should clear all data even if the network
     *                              request was unsuccessful, false otherwise.
     * @param handler               The handler that the callback will be invoked through.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult disconnect(final BasicResultCallback callback, final boolean disconnectOnException,
            @NonNull final Handler handler) {
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(callback, handler, executorService);
        executorService.execute(() -> {
            try {
                analytics.flush(null, false);

                // Call Disable push token instead of Logout, this will make sure push token is deleted
                Result disconnected = FlyAway.delete(
                        context,
                        DISABLE_PUSH,
                        "PushManager.disablePush",
                        null
                );
                if (disconnected.getStatus() == RequestStatus.COMPLETED || disconnectOnException) {
                    Utilities.clearSDKData(context, analytics);
                    sharedElements.setPushTokenStatus(SharedElements.KEY_IS_TOKEN_SENT_DELETED);
                    ScopeOfficer.INSTANCE.onDisconnected(context);
                    request.setResult(new Result(200, ""));
                } else {
                    sharedElements.setPushTokenStatus(SharedElements.KEY_IS_TOKEN_SENT_UNKNOWN);
                    // If an opted out user disconnects, needs to complete action formally clearing out
                    // SharePref and dbs also update user table.
                    if (disconnected.getStatus() == RequestStatus.OPTED_OUT) {
                        Utilities.clearSDKData(context, analytics);
                        ScopeOfficer.INSTANCE.onDisconnected(context);
                        request.setResult(new Result(200, ""));
                    } else {
                        if (disconnectOnException) {
                            Utilities.clearSDKData(context, analytics);
                            ScopeOfficer.INSTANCE.onDisconnected(context);
                        }
                        request.setFailed(disconnected.getException());
                    }
                }
            } catch (final FlybitsException e) {
                sharedElements.setPushTokenStatus(SharedElements.KEY_IS_TOKEN_SENT_UNKNOWN);
                if (disconnectOnException) {
                    Utilities.clearSDKData(context, analytics);
                    ScopeOfficer.INSTANCE.onDisconnected(context);
                }
                request.setFailed(e);
            }
        });
        return request;
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context)} method.
     *
     * @param callback The callback used to indicate whether or not the disconnection was
     *                 successful or not.
     * @param handler  The handler that the callback will be invoked through.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    BasicResult disconnect(final BasicResultCallback callback, @NonNull final Handler handler) {
        return disconnect(callback, false, handler);
    }

    /**
     * The {@code disconnect} method is responsible for clearing the server sessions with Flybits.
     * In addition, once a logout is successful it will notify all the registered
     * {@link FlybitsScope}s through the {@link FlybitsScope#onDisconnected(Context)} method.
     *
     * @param callback              The callback used to indicate whether or not the disconnection was
     *                              successful or not.
     * @param disconnectOnException true if the disconnect should clear all data even if the network
     *                              request was unsuccessful, false otherwise.
     *
     * @return The {@link BasicResult} that represents the network request for this query.
     */
    public BasicResult disconnect(final BasicResultCallback callback, final boolean disconnectOnException) {
        return disconnect(callback, disconnectOnException, new Handler(Looper.getMainLooper()));
    }

    /**
     * Check to see the status of given customer id with the saved jwt.
     *
     * @param customerId The unique identifier assigned by the client that represents user and it does not represent flybits user id.
     * @param callback   The {@link ObjectResultCallback} callback that indicates the status of the customerId based on connection
     *                   and opt in to Flybits Server.
     */
    public ObjectResult<CustomerStatus> checkCustomerIdStatus(@NonNull String customerId,
            final ObjectResultCallback<CustomerStatus> callback) {
        return checkCustomerIdStatus(customerId, callback, new Handler(Looper.getMainLooper()));
    }

    ObjectResult<CustomerStatus> checkCustomerIdStatus(@NonNull String customerId,
            final ObjectResultCallback<CustomerStatus> callback, Handler handler) {
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<CustomerStatus> query = new ObjectResult<>(callback, handler, executorService);
        String savedJwt = sharedElements.getSavedJWTToken();
        executorService.execute(() -> {
            UserDAO userDAO = CommonsDatabase.getDatabase(context).userDao();
            if (!savedJwt.isEmpty()) {
                Jwt jwt = Jwt.decodeJWTToken(savedJwt);
                long timeEpochTimeSeconds = new Date().getTime() / 1000;
                User user = userDAO.getSingleByUserAndCustomerId(jwt.getUserID(), customerId);
                setUserStatus(jwt, timeEpochTimeSeconds, user, query, customerId, userDAO);
            } else {
                query.setSuccess(CustomerStatus.NOT_CONNECTED);
            }
        });
        return query;
    }

    /**
     * Check to see the status of current user.
     *
     * @param callback The {@link ObjectResultCallback} callback that indicates the status of the customerId based on connection
     *                 and opt in to Flybits Server.
     */
    public ObjectResult<CustomerStatus> checkUserStatus(final ObjectResultCallback<CustomerStatus> callback) {
        return checkUserStatus(callback, new Handler(Looper.getMainLooper()));
    }

    ObjectResult<CustomerStatus> checkUserStatus(final ObjectResultCallback<CustomerStatus> callback, Handler handler) {
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<CustomerStatus> query = new ObjectResult<>(callback, handler, executorService);
        String savedJwt = sharedElements.getSavedJWTToken();
        executorService.execute(() -> {
            UserDAO userDAO = CommonsDatabase.getDatabase(context).userDao();
            if (!savedJwt.isEmpty()) {
                Jwt jwt = Jwt.decodeJWTToken(savedJwt);
                long timeEpochTimeSeconds = new Date().getTime() / 1000;
                User user = userDAO.getSingleById(jwt.getUserID());
                setUserStatus(jwt, timeEpochTimeSeconds, user, query, null, null);
            } else {
                query.setSuccess(CustomerStatus.NOT_CONNECTED);
            }
        });
        return query;
    }

    private void setUserStatus(Jwt jwt, Long timeEpochTimeSeconds, User user,
            ObjectResult<CustomerStatus> query, @Nullable String customerId,
            @Nullable UserDAO userDAO) {
        if (user != null) {
            if (user.isOptedIn() && (jwt.getExpiry() > timeEpochTimeSeconds)) {
                query.setSuccess(CustomerStatus.CONNECTED_OPTED_IN);
            } else if (user.isOptedIn() && (jwt.getExpiry() < timeEpochTimeSeconds)) {
                query.setSuccess(CustomerStatus.CONNECTED_OPTED_IN_EXPIRED_TOKEN);
            } else if (!user.isOptedIn() && (jwt.getExpiry() > timeEpochTimeSeconds)) {
                query.setSuccess(CustomerStatus.CONNECTED_OPTED_OUT);
            } else if (!user.isOptedIn() && (jwt.getExpiry() < timeEpochTimeSeconds)) {
                query.setSuccess(CustomerStatus.CONNECTED_OPTED_OUT_EXPIRED_TOKEN);
            }
        } else {
            if (customerId == null) {
                query.setSuccess(CustomerStatus.NOT_CONNECTED);
            } else {
                if (userDAO != null) {
                    user = userDAO.getSingleByCustomerId(customerId);
                    if (user != null) {
                        if (user.isOptedIn()) {
                            query.setSuccess(CustomerStatus.NOT_CONNECTED_OPTED_IN);
                        } else {
                            query.setSuccess(CustomerStatus.NOT_CONNECTED_OPTED_OUT);
                        }
                    } else {
                        query.setSuccess(CustomerStatus.NOT_CONNECTED);
                    }
                }
            }
        }
    }

    /**
     * @return The {@link User} with the current active connection.
     */
    public User getActiveUser(Context context) {
        if (isConnected(context)) {
            SharedElements sharedElements = SharedElementsFactory.INSTANCE.get(context);
            try {
                return sharedElements.getUser();
            } catch (Exception e) {
                return null;
            }
        }
        return null;
    }

    IDP getIDP() {
        return idProvider;
    }

    /**
     * The {@code Builder} class is responsible for setting all the attributes associated to the
     * {@link FlybitsManager}.
     */
    public static final class Builder {

        private final Set<FlybitsScope> listOfScopes;
        private Context mContext;
        private IDP idProvider;
        private ArrayList<String> languageCodes;
        private Boolean failedEncryptedToMemory = false;
        private Boolean retainSharedPref = true;

        /**
         * Constructor that initializes all the objects within the {@code Builder} class.
         *
         * @param context The {@link Context} of the application.
         */
        public Builder(@NonNull Context context) {
            this.mContext = context;
            this.listOfScopes = new HashSet<>();
            this.languageCodes = new ArrayList<>();
            try {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    final LocaleList localeList = LocaleList.getAdjustedDefault();
                    for (int i = 0; i < localeList.size(); i++) {
                        final Locale locale = localeList.get(i);
                        this.languageCodes.add(locale.getLanguage());
                    }
                } else {
                    this.languageCodes.add(Locale.getDefault().getLanguage());
                }
            } catch (final Exception e) {
                Logger.exception("FlybitsManager.Builder", e);
            }
            //Add new commonscope
            listOfScopes.add(new CommonScope(mContext));
        }

        /**
         * Build the {@link FlybitsManager} object that is used for defining the SDKs that are
         * associated to the application.
         *
         * @return {@link FlybitsManager} object which can be referenced by the SDK to get specific
         * information about the SDKs defined within the application.
         *
         * @throws InvalidFlybitsManagerException if the {@link Builder#addScope(FlybitsScope, Context)}
         *                                        method was not called.
         */
        public FlybitsManager build() throws InvalidFlybitsManagerException {
            checkIfFieldsSet();
            // Add scopes if called with builder.addScope()
            for (FlybitsScope listOfScope : listOfScopes) {
                FlybitsManager.addScope(listOfScope, mContext);
            }
            return new FlybitsManager(this);
        }

        /**
         * Sets the {@link IDP} which should be used sign into the Flybits Account Manager.
         *
         * @param idp The {@link IDP} that should be used for registering contextual information to
         *            the account.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setAccount(IDP idp) {
            this.idProvider = idp;
            return this;
        }

        /**
         * Sets the Language that the SDK should parse the localized values with.
         *
         * @param languageCode The 2-letter code, such as "en", that indicates which language should
         *                     be used for parsing localized values.
         *
         * @return A {@link Builder} object where additional {@link FlybitsManager} attributes can
         * be set.
         */
        public FlybitsManager.Builder setLanguage(@NonNull String languageCode) {
            languageCodes.clear();
            languageCodes.add(languageCode);
            return this;
        }

        private void checkIfFieldsSet() throws InvalidFlybitsManagerException {
            if (languageCodes.size() > 0 && languageCodes.get(0).length() != 2) {
                throw new InvalidFlybitsManagerException("Your language must be a 2-letter code. Make sure you call setLanguage(String)");
            }
        }

        /**
         * When the SharedPreference encryption are not supported by the device,
         * keep the sensitive data in memory
         *
         * @param memoryOnly true:  save values to the SP with private mode
         *                   false: sensitive data only save to memory.
         *                   this means the device will not remember authorization state in terms of sensitive data.
         *                   The App needs to re-authorized on next App launch
         */
        public FlybitsManager.Builder setFailedEncryptedToMemory(Boolean memoryOnly) {
            this.failedEncryptedToMemory = memoryOnly;
            return this;
        }

        /**
         * The method work with [failedEncryptedToMemory] is true and the encryption are not supported by the device;
         *
         * @param retainSharedPref true: clear the sensitive data in SP
         */
        public FlybitsManager.Builder setKeepSharedPref(Boolean retainSharedPref) {
            this.retainSharedPref = retainSharedPref;
            return this;
        }
    }
}
