/*
 * Decompiled with CFR 0.152.
 */
package com.feedzai.commons.sql.abstraction.batch;

import com.feedzai.commons.sql.abstraction.batch.BatchEntry;
import com.feedzai.commons.sql.abstraction.engine.DatabaseEngine;
import com.feedzai.commons.sql.abstraction.engine.DatabaseEngineException;
import com.feedzai.commons.sql.abstraction.entry.EntityEntry;
import com.google.common.base.Strings;
import java.util.LinkedList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.lang.time.DurationFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

public abstract class AbstractBatch
implements Runnable {
    protected final Logger logger = LoggerFactory.getLogger(AbstractBatch.class);
    protected static final Marker dev = MarkerFactory.getMarker((String)"DEV");
    protected static final int salt = 100;
    private final Lock flushLock = new ReentrantLock();
    protected final DatabaseEngine de;
    protected final long maxAwaitTimeShutdown;
    protected ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1);
    protected final int batchSize;
    protected final long batchTimeout;
    protected int batch;
    protected long lastFlush;
    protected LinkedList<BatchEntry> buffer = new LinkedList();
    protected String name;

    protected AbstractBatch(DatabaseEngine de, String name, int batchSize, long batchTimeout, long maxAwaitTimeShutdown) {
        this.de = de;
        this.batchSize = batchSize;
        this.batch = batchSize;
        this.batchTimeout = batchTimeout;
        this.lastFlush = System.currentTimeMillis();
        this.name = Strings.isNullOrEmpty((String)name) ? "Anonymous Batch" : name;
        this.maxAwaitTimeShutdown = maxAwaitTimeShutdown;
    }

    protected AbstractBatch(DatabaseEngine de, int batchSize, long batchTimeout, long maxAwaitTimeShutdown) {
        this(de, null, batchSize, batchTimeout, maxAwaitTimeShutdown);
    }

    protected void start() {
        this.scheduler.scheduleAtFixedRate(this, 0L, this.batchTimeout + 100L, TimeUnit.MILLISECONDS);
    }

    public synchronized void destroy() {
        this.logger.trace("{} - Destroy called on Batch", (Object)this.name);
        this.scheduler.shutdownNow();
        try {
            if (!this.scheduler.awaitTermination(this.maxAwaitTimeShutdown, TimeUnit.MILLISECONDS)) {
                this.logger.warn("Could not terminate batch within {}", (Object)DurationFormatUtils.formatDurationWords((long)this.maxAwaitTimeShutdown, (boolean)true, (boolean)true));
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.logger.debug("Interrupted while waiting.", (Throwable)e);
        }
        this.flush();
    }

    public void add(BatchEntry batchEntry) throws DatabaseEngineException {
        this.flushLock.lock();
        try {
            this.buffer.add(batchEntry);
            --this.batch;
        }
        finally {
            this.flushLock.unlock();
        }
        if (this.batch <= 0) {
            this.flush();
        }
    }

    public void add(String entityName, EntityEntry ee) throws DatabaseEngineException {
        this.add(new BatchEntry(entityName, ee));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void flush() {
        LinkedList<BatchEntry> temp;
        this.flushLock.lock();
        try {
            this.lastFlush = System.currentTimeMillis();
            if (this.batch == this.batchSize) {
                this.logger.trace("[{}] Batch empty, not flushing", (Object)this.name);
                return;
            }
            this.batch = this.batchSize;
            temp = this.buffer;
            this.buffer = new LinkedList();
        }
        finally {
            this.flushLock.unlock();
        }
        try {
            long start = System.currentTimeMillis();
            this.de.beginTransaction();
            for (BatchEntry entry : temp) {
                this.de.addBatch(entry.getTableName(), entry.getEntityEntry());
            }
            try {
                this.de.flush();
                this.de.commit();
                this.logger.trace("[{}] Batch flushed. Took {} ms, {} rows.", new Object[]{this.name, System.currentTimeMillis() - start, temp.size()});
            }
            finally {
                if (this.de.isTransactionActive()) {
                    this.de.rollback();
                }
            }
        }
        catch (Exception e) {
            this.logger.error(dev, "[{}] Error occurred while flushing.", (Object)this.name, (Object)e);
            this.onFlushFailure(temp.toArray(new BatchEntry[temp.size()]));
        }
    }

    public abstract void onFlushFailure(BatchEntry[] var1);

    @Override
    public synchronized void run() {
        if (System.currentTimeMillis() - this.lastFlush >= this.batchTimeout) {
            this.logger.trace("[{}] Flush timeout occurred", (Object)this.name);
            this.flush();
        }
    }
}

