/*
 * Decompiled with CFR 0.152.
 */
package com.feedzai.commons.sql.abstraction.batch;

import com.feedzai.commons.sql.abstraction.FailureListener;
import com.feedzai.commons.sql.abstraction.batch.BatchEntry;
import com.feedzai.commons.sql.abstraction.engine.DatabaseEngine;
import com.feedzai.commons.sql.abstraction.engine.DatabaseEngineException;
import com.feedzai.commons.sql.abstraction.entry.EntityEntry;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.util.LinkedList;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.lang.time.DurationFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

public abstract class AbstractBatch
implements Runnable {
    protected final Logger logger = LoggerFactory.getLogger(AbstractBatch.class);
    public static final FailureListener NO_OP = rowsFailed -> {};
    protected static final Marker dev = MarkerFactory.getMarker((String)"DEV");
    protected static final int salt = 100;
    private final Lock bufferLock = new ReentrantLock();
    private final Lock flushTransactionLock = new ReentrantLock();
    protected final DatabaseEngine de;
    protected final long maxAwaitTimeShutdown;
    protected ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(1);
    protected final int batchSize;
    protected final long batchTimeout;
    protected int batch;
    protected volatile long lastFlush;
    protected LinkedList<BatchEntry> buffer = new LinkedList();
    protected String name;
    protected Optional<FailureListener> failureListener = Optional.empty();

    protected AbstractBatch(DatabaseEngine de, String name, int batchSize, long batchTimeout, long maxAwaitTimeShutdown, FailureListener failureListener) {
        Preconditions.checkNotNull((Object)de, (Object)"The provided database engine is null.");
        Preconditions.checkNotNull((Object)failureListener, (Object)"The provided failure listener is null");
        this.de = de;
        this.batchSize = batchSize;
        this.batch = batchSize;
        this.batchTimeout = batchTimeout;
        this.lastFlush = System.currentTimeMillis();
        this.name = Strings.isNullOrEmpty((String)name) ? "Anonymous Batch" : name;
        this.maxAwaitTimeShutdown = maxAwaitTimeShutdown;
        this.failureListener = Optional.of(failureListener);
    }

    protected AbstractBatch(DatabaseEngine de, String name, int batchSize, long batchTimeout, long maxAwaitTimeShutdown) {
        this(de, name, batchSize, batchTimeout, maxAwaitTimeShutdown, NO_OP);
    }

    protected AbstractBatch(DatabaseEngine de, int batchSize, long batchTimeout, long maxAwaitTimeShutdown) {
        this(de, null, batchSize, batchTimeout, maxAwaitTimeShutdown);
    }

    protected void start() {
        this.scheduler.scheduleAtFixedRate(this, 0L, this.batchTimeout + 100L, TimeUnit.MILLISECONDS);
    }

    public void destroy() {
        this.logger.trace("{} - Destroy called on Batch", (Object)this.name);
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(this.maxAwaitTimeShutdown, TimeUnit.MILLISECONDS)) {
                this.logger.warn("Could not terminate batch within {}. Forcing shutdown.", (Object)DurationFormatUtils.formatDurationWords((long)this.maxAwaitTimeShutdown, (boolean)true, (boolean)true));
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            this.logger.debug("Interrupted while waiting.", (Throwable)e);
        }
        this.flush();
    }

    public void add(BatchEntry batchEntry) throws DatabaseEngineException {
        this.bufferLock.lock();
        try {
            this.buffer.add(batchEntry);
            --this.batch;
        }
        finally {
            this.bufferLock.unlock();
        }
        if (this.batch <= 0) {
            this.flush();
        }
    }

    public void add(String entityName, EntityEntry ee) throws DatabaseEngineException {
        this.add(new BatchEntry(entityName, ee));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void flush() {
        LinkedList<BatchEntry> temp;
        this.bufferLock.lock();
        try {
            this.lastFlush = System.currentTimeMillis();
            if (this.batch == this.batchSize) {
                this.logger.trace("[{}] Batch empty, not flushing", (Object)this.name);
                return;
            }
            this.batch = this.batchSize;
            temp = this.buffer;
            this.buffer = new LinkedList();
        }
        finally {
            this.bufferLock.unlock();
        }
        try {
            this.flushTransactionLock.lock();
            long start = System.currentTimeMillis();
            this.de.beginTransaction();
            for (BatchEntry entry : temp) {
                this.de.addBatch(entry.getTableName(), entry.getEntityEntry());
            }
            this.de.flush();
            this.de.commit();
            this.logger.trace("[{}] Batch flushed. Took {} ms, {} rows.", new Object[]{this.name, System.currentTimeMillis() - start, temp.size()});
        }
        catch (Exception e) {
            this.logger.error(dev, "[{}] Error occurred while flushing.", (Object)this.name, (Object)e);
            this.onFlushFailure(temp.toArray(new BatchEntry[temp.size()]));
        }
        finally {
            try {
                if (this.de.isTransactionActive()) {
                    this.de.rollback();
                }
            }
            catch (Exception e) {
                this.logger.trace("[{}] Batch failed to check the flush transaction state", (Object)this.name);
            }
            this.flushTransactionLock.unlock();
        }
    }

    public void flush(boolean sync) {
        if (!sync) {
            this.flush();
        } else {
            try {
                this.flushTransactionLock.lock();
                this.flush();
            }
            finally {
                this.flushTransactionLock.unlock();
            }
        }
    }

    public void onFlushFailure(BatchEntry[] entries) {
        if (!this.failureListener.isPresent()) {
            return;
        }
        this.failureListener.get().onFailure(entries);
    }

    @Override
    public void run() {
        if (System.currentTimeMillis() - this.lastFlush >= this.batchTimeout) {
            this.logger.trace("[{}] Flush timeout occurred", (Object)this.name);
            this.flush();
        }
    }
}

