/*
 *  Copyright 2017 Expedia, Inc.
 *
 *       Licensed under the Apache License, Version 2.0 (the "License");
 *       you may not use this file except in compliance with the License.
 *      You may obtain a copy of the License at
 *
 *           http://www.apache.org/licenses/LICENSE-2.0
 *
 *       Unless required by applicable law or agreed to in writing, software
 *       distributed under the License is distributed on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *       See the License for the specific language governing permissions and
 *       limitations under the License.
 */

package com.expedia.www.haystack.trace.reader.readers.transformers

import java.util.UUID

import com.expedia.open.tracing.{Span, Tag}
import com.expedia.www.haystack.trace.reader.readers.utils.{MutableSpanForest, SpanUtils}

/**
  *
  * If there are multiple roots in the given trace, use the first root based on startTime to be root
  * mark other roots as children of the selected root
  * If there is no root, assume loopback span or first span in time order to be root
  *
  * **Apply this transformer only if you are not confident about clients sending in roots properly**
  */
class InvalidRootTransformer extends SpanTreeTransformer {
  private val AUTOGEN_REASON =
    """
      |This span is autogenerated by haystack and only a UI sugar to show multiple root spans together in one view.
      | This is a symptom that few spans have empty parent id, but only one such root span should exist.
    """.stripMargin

  override def transform(spanForest: MutableSpanForest): MutableSpanForest = {
    val rootSpans = spanForest.getAllTrees.filter(_.span.getParentSpanId.isEmpty).map(_.span)

    rootSpans.size match {
      case 0 => toTraceWithAssumedRoot(spanForest)
      case 1 => spanForest
      case _ => toTraceWithSingleRoot(spanForest, rootSpans.size)
    }
  }

  private def toTraceWithAssumedRoot(forest: MutableSpanForest): MutableSpanForest = {
    // if we have just one tree, then simply set it's root's parent spanId as empty
    if (forest.countTrees <= 1) {
      return forest.updateEachSpanTreeRoot(resetParentSpanId)
    }

    // if we have just 1 loopback tree root, which means its parentSpanId is the same as its spanId, then mark it as a root
    // by setting its parentSpanId as empty
    val loopbackTrees = forest.treesWithLoopbackRoots
    if (loopbackTrees.size == 1) {
      return forest.updateEachSpanTreeRoot((span) => if (loopbackTrees.head.span == span) resetParentSpanId(span) else span)
    }

    // for all other cases, get the root with the minimum startTime and make it as a root by setting parentSpanId as empty
    val spanWithMinStartTime = forest.getAllTrees.minBy(_.span.getStartTime).span
    forest.updateEachSpanTreeRoot((span) => if (span == spanWithMinStartTime) resetParentSpanId(span) else span)
  }

  private def toTraceWithSingleRoot(forest: MutableSpanForest, emptyParentIdSpanTrees: Int): MutableSpanForest = {
    val allTreeRootSpans: Seq[Span] = forest.getAllTrees.map(_.span)
    val newRootSpan = SpanUtils.addClientLogTag(SpanUtils
      .createAutoGeneratedRootSpan(allTreeRootSpans, AUTOGEN_REASON, UUID.randomUUID().toString)
      .addTags(Tag.newBuilder().setKey("X-HAYSTACK-SPAN-ROOT-COUNT").setVLong(emptyParentIdSpanTrees).setType(Tag.TagType.LONG))
      .build())

    forest.addNewRoot(newRootSpan)
  }

  private def resetParentSpanId(span: Span): Span = Span.newBuilder(span).setParentSpanId("").build()
}
