/*
 * Copyright (C) 2017 - present by PyXLL Ltd.
 */
package com.exceljava.jinx;

import java.io.File;
import java.util.List;
import java.util.logging.LogRecord;
import java.lang.ClassLoader;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.lang.Runnable;

/**
 * Excel Addin
 * <p>
 * Encapsulates the methods exposed by the Excel addin to Java
 * when embedded in Excel.
 *
 * Classes loaded by Jinx with a public constructor taking
 * a single {@link com.exceljava.jinx.ExcelAddIn} parameter
 * will be instantiated with the {@link com.exceljava.jinx.ExcelAddIn}
 * instance when the add-in starts.
 */
public interface ExcelAddIn {
    /**
     * Gets the full path to the .xll add-in file.
     * @return Path to .xll add-in file.
     */
    public abstract String getAddInPath();

    /**
     * Gets full path to the log file.
     * @return Path to log file.
     */
    public abstract String getLogPath();

    /**
     * Gets full path to the config file.
     * @return Path to config file.
     */
    public abstract String getConfigPath();

    /**
     * Gets the custom class loaded used for loading classes
     * listed on the class path in the config.
     *
     * Jinx uses two different class loaders. One for reloadable
     * classes (the default), and another for non-reloadable
     * classes.
     *
     * This method returns the reloadable class loader, which will
     * change after reloading the addin.
     *
     * @return Current reloadable class loader.
     */
    public abstract ClassLoader getClassLoader();

    /**
     * Gets custom class loaded used for loading classes
     * listed on the class path in the config.
     * *
     * Jinx uses two different class loaders. One for reloadable
     * classes (the default), and another for non-reloadable
     * classes.
     *
     * The reloadable class loader is replaced when the Jinx
     * add-in is reloaded, but the non-reloadable one isn't.
     *
     * @param nonReloadable Return the non-reloadable ClassLoader if true.
     * @return Current reloadable or non-reloadable class loader.
     * @since 1.5
     */
    public abstract ClassLoader getClassLoader(boolean nonReloadable);

    /**
     * The Jinx class loaders copy jars and class files to a
     * shadow folder, if the 'shadow_path' is set.
     *
     * This is so that the original JARs aren't locked by the JVM
     * and can be overwritten before reloading.
     *
     * This method returns the path where the current reloadable
     * class loader copies jar and class files to.
     *
     * @return Path used by the current reloadable class loader.
     * @since 1.5
     */
    public abstract File getShadowPath();

    /**
     * The Jinx class loaders copy jars and class files to a
     * shadow folder, if the 'shadow_path' is set.
     *
     * This is so that the original JARs aren't locked by the JVM
     * and can be overwritten before reloading.
     *
     * This method returns the path where the current reloadable and
     * non-reloadable class loaders copy jar and class files to.
     *
     * @param nonReloadable Return the non-reloadable ClassLoader if true.
     * @return Path used by the current reloadable or non-reloadable class loader.
     * @since 1.5
     */
    public abstract File getShadowPath(boolean nonReloadable);

    /**
     * Gets the set of config section names.
     *
     * @return Array of config section names
     */
    public abstract List<String> getConfigSections();

    /**
     * Gets the set of keys in a config section.
     * @param section The section to get the keys of.
     * @return Array of config section keys.
     */
    public abstract List<String> getConfigSectionKeys(String section);

    /**
     * Tests if the config contains a specific key.
     * @param section Section key belongs to.
     * @param key Name of the key to check for.
     * @return true if the config contains a value for the section and key.
     */
    public abstract boolean hasConfigValue(String section, String key);

    /**
     * Retrieves a value from the config.
     * @param section Section to retrieve the value of key from.
     * @param key Key to retrieve the value for.
     * @return Value of the key in the section.
     */
    public abstract String getConfigValue(String section, String key);

    /**
     * Convert an argument to an Excel function to another Java type
     * using any standard conversions or custom argument converters
     * registered using ExcelArgumentConverter.
     * @param arg Object passed from Excel to convert.
     * @param cls Class to try to convert the argument to.
     * @return The argument converted to the target class.
     */
    public abstract <T> T convertArgument(Object arg, Class<T> cls);

    /**
     * Convert a value to be returned to Excel from any Java type to
     * a type that can be displayed in Excel using any standard conversions
     * or custom return converters registered using ExcelReturnConverter.
     * @param value Object to be returned to Excel.
     * @return The Object converted to the Excel type.
     */
    public abstract Object convertReturnValue(Object value);

    /**
     * Retrieves the COM Excel.Application instance for the current
     * Excel process.
     *
     * The returned long is a new reference to an IUnknown C pointer.
     * The caller must call IUnknown::Release on it from the same
     * thread to avoid a resource leak.
     *
     * This is a low level call only intended for supporting libraries
     * and shouldn't need to be called by user code.
     *
     * @return Pointer to new reference of an IUnknown COM object.
     */
    public abstract long getExcelApplication();

    /**
     * Retrieves the COM Excel.Application instance for the current
     * Excel process, converted to a COM wrapper class.
     *
     * There must be a registered converter for converting from an
     * IUnknown object to the wrapper class.
     *
     * The jinx-com4j project provides a set of Excel wrapper classes that
     * can be used with this method.
     *
     * @return Excel.Application COM wrapper object.
     */
    public abstract <T> T getExcelApplication(Class<T> cls);

    /**
     * Logs to the xll log file.
     * @param logRecord Log record to write to the log.
     */
    public abstract void log(LogRecord logRecord);

    /**
     * Triggers a reload of the Jinx Excel Addin.
     * Reloading doesn't occur until after control has
     * returned from Java to Excel.
     */
    public abstract void reload();

    /**
     * Show the 'about' dialog box.
     */
    public abstract void about();

    /**
     * Notify Excel that an Rtd value has been updated.
     * @param rtd Rtd instance that has updated.
     */
    public abstract void notify(Rtd rtd);

    /**
     * Scan a class for {@link com.exceljava.jinx.ExcelFunction},
     * {@link com.exceljava.jinx.ExcelMacro} and {@link com.exceljava.jinx.ExcelMenu}
     * annotations.
     *
     * If the class has any annotated pubic instance methods, an instance
     * will be constructed using the default constructor.
     *
     * After scanning a class or classes, {@link com.exceljava.jinx.ExcelAddIn#rebind}
     * must be called to create the Excel bindings for the discovered functions.
     *
     * @param cls Class to scan for annotations.
     */
    public abstract void scan(Class cls);

    /**
     * Scan a class for {@link com.exceljava.jinx.ExcelFunction},
     * {@link com.exceljava.jinx.ExcelMacro} and {@link com.exceljava.jinx.ExcelMenu}
     * annotations.
     *
     * If the class has any annotated pubic instance methods, the instance
     * passed to this method will be used when invoking those methods.
     *
     * After scanning a class or classes, {@link com.exceljava.jinx.ExcelAddIn#rebind}
     * must be called to create the Excel bindings for the discovered functions.
     *
     * @param cls Class to scan for annotations.
     * @param instance Instance to use for binding instance methods.
     * @since 1.5
     */
    public abstract <T> void scan(Class<T> cls, T instance);

    /**
     * Scan a class for {@link com.exceljava.jinx.ExcelFunction},
     * {@link com.exceljava.jinx.ExcelMacro} and {@link com.exceljava.jinx.ExcelMenu}
     * annotations.
     *
     * If the class has any annotated pubic instance methods, an instance
     * will be constructed using the default constructor.
     *
     * The ClassLoader is set as the thread's context class loader before
     * any annotated methods are invoked. This can be useful if you are
     * using multiple class loaders to isolate sets of functionality
     * in your add-in.
     *
     * After scanning a class or classes, {@link com.exceljava.jinx.ExcelAddIn#rebind}
     * must be called to create the Excel bindings for the discovered functions.
     *
     * @param cls Class to scan for annotations.
     * @param classLoader ClassLoader used to load the class.
     * @since 1.5
     */
    public abstract void scan(Class cls, ClassLoader classLoader);

    /**
     * Scan a class for {@link com.exceljava.jinx.ExcelFunction},
     * {@link com.exceljava.jinx.ExcelMacro} and {@link com.exceljava.jinx.ExcelMenu}
     * annotations.
     *
     * If the class has any annotated pubic instance methods, the instance
     * passed to this method will be used when invoking those methods.
     *
     * The ClassLoader is set as the thread's context class loader before
     * any annotated methods are invoked. This can be useful if you are
     * using multiple class loaders to isolate sets of functionality
     * in your add-in.
     *
     * After scanning a class or classes, {@link com.exceljava.jinx.ExcelAddIn#rebind}
     * must be called to create the Excel bindings for the discovered functions.
     *
     * @param cls Class to scan for annotations.
     * @param classLoader ClassLoader used to load the class.
     * @param instance Instance to use for binding instance methods.
     * @since 1.5
     */
    public abstract <T> void scan(Class<T> cls, ClassLoader classLoader, T instance);

    /**
     * Find all resources named "jinx-classes.txt" using the class loader.
     * Any classes listed in the resources are scanned for
     * {@link com.exceljava.jinx.ExcelFunction}, {@link com.exceljava.jinx.ExcelMacro}
     * and {@link com.exceljava.jinx.ExcelMenu} annotations.
     *
     * After scanning a class or classes, {@link com.exceljava.jinx.ExcelAddIn#rebind}
     * must be called to create the Excel bindings for the discovered functions.
     *
     * @param classLoader ClassLoader used to look for resources and load the classes.
     * @since 1.5
     */
    public abstract void scan(ClassLoader classLoader);

    /**
     * Find all resources with a given resource name using the class loader.
     * Any classes listed in the resources are scanned for
     * {@link com.exceljava.jinx.ExcelFunction}, {@link com.exceljava.jinx.ExcelMacro}
     * and {@link com.exceljava.jinx.ExcelMenu} annotations.
     *
     * After scanning a class or classes, {@link com.exceljava.jinx.ExcelAddIn#rebind}
     * must be called to create the Excel bindings for the discovered functions.
     *
     * @param classLoader ClassLoader used to look for resources and load the classes.
     * @since 1.5
     */
    public abstract void scan(ClassLoader classLoader, String resourceName);

    /**
     * Schedule the Jinx add-in to rebuild the bindings between the annotated
     * Java methods and Excel. The actual re-binding will occur shortly after
     * control has returned to Excel.
     *
     * This should be called after calls to {@link com.exceljava.jinx.ExcelAddIn#scan}
     * to finish exposing any annotated methods to Excel as UDFs, menus or macros.
     *
     * Any instances constructed during {@link com.exceljava.jinx.ExcelAddIn#scan}
     * will not be re-instantiated, it is only the binding between the classes
     * and any previously constructed instances and Excel that is refreshed.
     */
    public abstract void rebind();

    /**
     * Get the number of objects in the object cache.
     *
     * @return number of cached objects.
     * @since 1.5
     */
    public abstract int getCachedObjectsCount();

    /**
     * Schedule a task to run in the main Excel thread in an Excel macro
     * context.
     *
     * Runnables called this way may call back into Excel using COM.
     *
     * @param task Runnable to run in the future.
     * @param delay Minimum delay before running the task.
     * @param units Units for the delay..
     * @since 1.6
     */
    public abstract Future schedule(Runnable task, long delay, TimeUnit units);

    /**
     * Schedule a task to run in the main Excel thread in an Excel macro
     * context as soon as possible.
     *
     * Runnables called this way may call back into Excel using COM.
     *
     * @param task Runnable to run in the future.
     * @since 1.6
     */
    public abstract Future schedule(Runnable task);

    /**
     * Can be called from an @ExcelFunction UDF.
     *
     * Returns a reference to the cells (or cells) calling the
     * function.
     *
     * @since 1.6
     */
    public abstract ExcelReference getCaller();

    /**
     * Returns the ribbon xml docuemnt as a String.
     *
     * @since 2.0
     */
    public abstract String getRibbonXml();

    /**
     * Sets the ribbon xml docuemnt as a String.
     * Calling this will cause the Excel ribbon to be recreated.
     *
     * @param xml Xml document containing the Excel ribbon customizations.
     * @since 2.0
     */
    public abstract void setRibbonXml(String xml);

    /**
     * Sets the ribbon xml docuemnt as a String.
     * Calling this will cause the Excel ribbon to be recreated.
     *
     * This overload can be used if the ClassLoader used for loading resources
     * (e.g. images) needs to be set differently from the main class loader.
     *
     * @param xml Xml document containing the Excel ribbon customizations.
     * @param classLoader ClassLoader used for loading resources (e.g. images).
     * @since 2.0
     */
    public abstract void setRibbonXml(String xml, ClassLoader classLoader);

    /**
     * Creates an ExcelReference instance from a Range COM object.
     *
     * @param range Excel Range COM object
     * @return Range reference as an ExcelReference instance.
     */
    public abstract ExcelReference getReference(IUnknown range);

    /**
     * Fetches the last value cached for a particular cell.
     * If no value is found, null is returned.
     *
     * @param cell Cell reference to fetch the object for.
     * @return Cached object, or null if not found.
     */
    public abstract Object getCachedObject(ExcelReference cell);

    /**
     * Fetches the last value cached for a particular cell.
     * If no value is found, null is returned.
     *
     * @param cell Cell reference to fetch the object for.
     * @param cls Expected class of the cached value.
     * @return Cached object, or null if not found.
     */
    public abstract <T> T getCachedObject(ExcelReference cell, Class<T> cls);

    /**
     * Sets the error handler to use to process exceptions raised by
     * functions, macros, menus and actions.
     *
     * @param errorHandler ErrorHandler to set.
     * @since 2.0
     */
    public abstract void setErrorHandler(ErrorHandler errorHandler);

    /**
     * Get the error handler used for handling Jinx function, macro
     * menu and action errors.
     *
     * @return ErrorHandler instance.
     * @since 2.0
     */
    public abstract ErrorHandler getErrorHandler();

    /**
     * Get the default error handler used for handling Jinx function, macro
     * menu and action errors.
     * May be used when implementing a custom error handler.
     *
     * @return ErrorHandler instance.
     * @since 2.0
     */
    public abstract ErrorHandler getDefaultErrorHandler();
}
