package com.exceljava.jinx;

import java.lang.reflect.Method;

/**
 * <p>ErrorHandler interface for handling errors thrown when calling Excel
 * functoins, macros, menus and ribbon actions.</p>
 *
 * <p>A custom error handler can be configured in the jinx.ini file,
 * for example:</p>
 *
 * <pre><code>
 * [JINX]
 * error_handler = com.mycompany.myapp.ErrorHandler
 * </code></pre>
 *
 * @since 2.0
 */
public interface ErrorHandler {
    /**
     * Called when an exception is thrown when calling a worksheet function
     * or UDF.
     *
     * @param functionName Excel name of the function called.
     * @param method Java method mapped to the function.
     * @param error Exception thrown.
     * @return Object to be returned to Excel. If null, an Excel error code will be used.
     */
    Object functionError(String functionName, Method method, Throwable error);

    /**
     * Called when an exception is thrown when calling a macro.
     *
     * @param macroName Excel name of the macro called.
     * @param method Java method mapped to the function.
     * @param error Exception thrown.
     * @return Object to be returned to Excel. If null, an Excel error code will be used.
     */
    Object macroError(String macroName, Method method, Throwable error);

    /**
     * Called when an exception is thrown calling a menu function.
     *
     * @param menuName Name of the menu containing the menu item.
     * @param subMenuName Name of the sub-menu, or null.
     * @param itemName Name of the menu item.
     * @param method Java method mapped to the menu item.
     * @param error Exception thrown.
     */
    void menuError(String menuName, String subMenuName, String itemName, Method method, Throwable error);

    /**
     * Called when an exception is thrown calling an action.
     *
     * @param actionName Name of the action called.
     * @param method Java method mapped to the menu item.
     * @param error Exception thrown.
     */
    void actionError(String actionName, Method method, Throwable error);
}
