package com.enterprisemath.utils.messaging;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;

import com.enterprisemath.utils.DomainUtils;
import com.enterprisemath.utils.ValidationUtils;
import java.util.ArrayList;
import java.util.List;

/**
 * Defines message. Message contains everything which needs to be there
 * to deliver full content to all recipients.
 *
 * @author radek.hecl
 *
 */
public class Message {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Message type.
         */
        private String type;

        /**
         * Parameters.
         */
        private Map<String, String> parameters = new HashMap<String, String>();

        /**
         * Attachments.
         */
        private List<Attachment> attachments = new ArrayList<Attachment>();

        /**
         * Sets type.
         *
         * @param type type
         * @return this instance
         */
        public Builder setType(String type) {
            this.type = type;
            return this;
        }

        /**
         * Sets parameters.
         *
         * @param parameters parameters
         * @return this instance
         */
        public Builder setParameters(Map<String, String> parameters) {
            this.parameters = DomainUtils.softCopyMap(parameters);
            return this;
        }

        /**
         * Adds parameter.
         *
         * @param key parameter key
         * @param value parameter value
         * @return this instance
         */
        public Builder addParameter(String key, String value) {
            parameters.put(key, value);
            return this;
        }

        /**
         * Sets attachments.
         *
         * @param attachments attachments
         * @return this instance
         */
        public Builder setAttachments(List<Attachment> attachments) {
            this.attachments = DomainUtils.softCopyList(attachments);
            return this;
        }

        /**
         * Adds attachment.
         *
         * @param attachment attachment
         * @return this instance
         */
        public Builder addAttachment(Attachment attachment) {
            attachments.add(attachment);
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public Message build() {
            return new Message(this);
        }
    }

    /**
     * Message type.
     */
    private String type;

    /**
     * Parameters.
     */
    private Map<String, String> parameters;

    /**
     * Attachments.
     */
    private List<Attachment> attachments;

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public Message(Builder builder) {
        type = builder.type;
        parameters = DomainUtils.softCopyUnmodifiableMap(builder.parameters);
        attachments = DomainUtils.softCopyUnmodifiableList(builder.attachments);
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardNotEmpty(type, "type cannot be empty");
        ValidationUtils.guardNotEmptyStringInCollection(parameters.keySet(), "parameters.keySet cannot have an empty string");
        ValidationUtils.guardNotNull(parameters.values(), "values cannot have null element");
        ValidationUtils.guardNotNullCollection(attachments, "attachements cannot have null element");
    }

    /**
     * Returns type.
     *
     * @return type
     */
    public String getType() {
        return type;
    }

    /**
     * Returns parameters.
     *
     * @return parameters
     */
    public Map<String, String> getParameters() {
        return parameters;
    }

    /**
     * Returns attachments.
     *
     * @return attachments
     */
    public List<Attachment> getAttachments() {
        return attachments;
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.reflectionHashCode(this);
    }

    @Override
    public boolean equals(Object obj) {
        return EqualsBuilder.reflectionEquals(this, obj);
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

}
