package com.enterprisemath.utils.image;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;

import org.apache.commons.lang3.builder.ToStringBuilder;

import com.enterprisemath.utils.ValidationUtils;

/**
 * Implementation of image animation which just shows constant color in all frames in all space.
 *
 * @author radek.hecl
 *
 */
public class ConstantColorImageAnimation implements ImageAnimation {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Frame width.
         */
        private Integer frameWidth;

        /**
         * Frame height.
         */
        private Integer frameHeight;

        /**
         * Frame duration in milliseconds.
         */
        private Integer frameDuration;

        /**
         * Number of frames.
         */
        private Integer numFrames;

        /**
         * Color used for the background.
         */
        private Color color;

        /**
         * Sets frame width.
         *
         * @param frameWidth frame width
         * @return this instance
         */
        public Builder setFrameWidth(Integer frameWidth) {
            this.frameWidth = frameWidth;
            return this;
        }

        /**
         * Sets frame height.
         *
         * @param frameHeight frame height
         * @return this instance
         */
        public Builder setFrameHeight(Integer frameHeight) {
            this.frameHeight = frameHeight;
            return this;
        }

        /**
         * Sets frame duration in milliseconds.
         *
         * @param frameDuration frame duration in milliseconds
         * @return this instance
         */
        public Builder setFrameDuration(Integer frameDuration) {
            this.frameDuration = frameDuration;
            return this;
        }

        /**
         * Sets number of frames.
         *
         * @param numFrames number of frames
         * @return this instance
         */
        public Builder setNumFrames(Integer numFrames) {
            this.numFrames = numFrames;
            return this;
        }

        /**
         * Sets color.
         *
         * @param color color
         * @return this instance
         */
        public Builder setColor(Color color) {
            this.color = color;
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public ConstantColorImageAnimation build() {
            return new ConstantColorImageAnimation(this);
        }
    }

    /**
     * Frame width.
     */
    private Integer frameWidth;

    /**
     * Frame height.
     */
    private Integer frameHeight;

    /**
     * Frame duration in milliseconds.
     */
    private Integer frameDuration;

    /**
     * Number of frames.
     */
    private Integer numFrames;

    /**
     * Color used for the background.
     */
    private Color color;

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public ConstantColorImageAnimation(Builder builder) {
        frameWidth = builder.frameWidth;
        frameHeight = builder.frameHeight;
        frameDuration = builder.frameDuration;
        numFrames = builder.numFrames;
        color = builder.color;
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardPositiveInt(frameWidth, "frameWidth must be positive");
        ValidationUtils.guardPositiveInt(frameHeight, "frameHeight must be positive");
        ValidationUtils.guardPositiveInt(frameDuration, "frameDuration must be positive");
        ValidationUtils.guardNotNegativeInt(numFrames, "numFrames cannot be negative");
        ValidationUtils.guardNotNull(color, "color cannot be null");
    }

    @Override
    public int getFrameWidth() {
        return frameWidth;
    }

    @Override
    public int getFrameHeight() {
        return frameHeight;
    }

    @Override
    public int getNumFrames() {
        return numFrames;
    }

    @Override
    public int getFrameDuration() {
        return frameDuration;
    }

    @Override
    public RenderedImage getFrame(int index) {
        BufferedImage img = new BufferedImage(frameWidth, frameHeight, BufferedImage.TYPE_4BYTE_ABGR);
        Graphics g = img.getGraphics();
        g.setColor(color);
        g.fill3DRect(0, 0, frameWidth + 1, frameHeight + 1, false);
        return img;
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

    /**
     * Creates new instance.
     *
     * @param frameWidth frame width
     * @param frameHeight frame height
     * @param frameDuration duration of one frame
     * @param numFrames number of frames
     * @param color color
     * @return created object
     */
    public static ConstantColorImageAnimation create(int frameWidth, int frameHeight, int frameDuration,
            int numFrames, Color color) {
        return new ConstantColorImageAnimation.Builder().
                setFrameWidth(frameWidth).
                setFrameHeight(frameHeight).
                setFrameDuration(frameDuration).
                setNumFrames(numFrames).
                setColor(color).
                build();
    }
}
