package com.enterprisemath.utils.engine;

import com.enterprisemath.utils.DomainUtils;
import com.enterprisemath.utils.ValidationUtils;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Defines order to run the task.
 *
 * @author radek.hecl
 */
public class TaskRunOrder {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Task class.
         */
        private String taskClass;

        /**
         * Parameters.
         */
        private Map<String, String> parameters = new HashMap<String, String>();

        /**
         * Sets task class.
         *
         * @param taskClass task class
         * @return this instance
         */
        public Builder setTaskClass(String taskClass) {
            this.taskClass = taskClass;
            return this;
        }

        /**
         * Sets parameters.
         *
         * @param parameters parameters
         * @return this instance
         */
        public Builder setParameters(Map<String, String> parameters) {
            this.parameters = DomainUtils.softCopyMap(parameters);
            return this;
        }

        /**
         * Adds parameters.
         *
         * @param key key
         * @param value value
         * @return this instance
         */
        public Builder addParameter(String key, String value) {
            this.parameters.put(key, value);
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public TaskRunOrder build() {
            return new TaskRunOrder(this);
        }
    }

    /**
     * Task class.
     */
    private String taskClass;

    /**
     * Parameters.
     */
    private Map<String, String> parameters;

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public TaskRunOrder(Builder builder) {
        taskClass = builder.taskClass;
        parameters = DomainUtils.softCopyUnmodifiableMap(builder.parameters);
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardNotEmpty(taskClass, "taskClass cannot be empty");
        ValidationUtils.guardNotEmptyNullMap(parameters, "parameters cannot have empty key or null value");
    }

    /**
     * Returns task class.
     *
     * @return task class
     */
    public String getTaskClass() {
        return taskClass;
    }

    /**
     * Returns parameters.
     *
     * @return parameters
     */
    public Map<String, String> getParameters() {
        return parameters;
    }

    /**
     * Creates instance of the task.
     *
     * @return created instance
     */
    public Task createTaskInstance() {
        try {
            Class<Task> clazz = (Class<Task>) Class.forName(taskClass);
            Method method = clazz.getMethod("create");
            return (Task) method.invoke(null);
        } catch (ClassNotFoundException e) {
            throw new RuntimeException(e);
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        } catch (SecurityException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e);
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.reflectionHashCode(this);
    }

    @Override
    public boolean equals(Object obj) {
        return EqualsBuilder.reflectionEquals(this, obj);
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

    /**
     * Creates new instance.
     *
     * @param taskClass task class
     * @return created instance
     */
    public static TaskRunOrder create(String taskClass) {
        return new TaskRunOrder.Builder().
                setTaskClass(taskClass).
                build();
    }

    /**
     * Creates new instance.
     *
     * @param taskClass task class
     * @param parameters parameters
     * @return created instance
     */
    public static TaskRunOrder create(String taskClass, Map<String, String> parameters) {
        return new TaskRunOrder.Builder().
                setTaskClass(taskClass).
                setParameters(parameters).
                build();
    }
}
