package com.enterprisemath.utils.engine;

import com.enterprisemath.utils.Dates;
import org.apache.commons.lang.builder.ToStringBuilder;

import com.enterprisemath.utils.DomainUtils;
import com.enterprisemath.utils.ValidationUtils;
import java.util.HashSet;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Implementation of task run listener which logs the result.
 * To be able use this listener the SLF4j dependency must be included in the project.
 *
 * @author radek.hecl
 */
public class Slf4jLogTaskRunListener implements TaskRunListener {

    /**
     * Log object.
     */
    private static final Logger log = LoggerFactory.getLogger(Slf4jLogTaskRunListener.class);

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Successful tasks which are pushed to the debug mode.
         */
        private Set<String> debugSuccessfulTasks = new HashSet<String>();

        /**
         * Format for the message with success.
         */
        private String successMessageFormat = "Task successfully finished: {code} - {taskClass} - {parameters} - {duration}";

        /**
         * Format for the message with exception.
         */
        private String exceptionMessageFormat = "Task error: {code} - {taskClass} - {parameters} - {exceptionDetails}";

        /**
         * Format for the message with stop.
         */
        private String stopMessageFormat = "Task was stopped: {code} - {taskClass} - {parameters}";

        /**
         * Sets successful tasks which are pushed to the debug mode.
         *
         * @param debugSuccessfulTasks successful tasks which are pushed to the debug mode
         * @return this instance
         */
        public Builder setDebugSuccessfulTasks(Set<String> debugSuccessfulTasks) {
            this.debugSuccessfulTasks = debugSuccessfulTasks;
            return this;
        }

        /**
         * Sets format for tasks which finishes by success.
         * Property access is done by '{path}' where path refers to the property in the report object.
         *
         * @param successMessageFormat message format
         * @return this instance
         */
        public Builder setSuccessMessageFormat(String successMessageFormat) {
            this.successMessageFormat = successMessageFormat;
            return this;
        }

        /**
         * Sets format for tasks which finishes by exception.
         * Property access is done by '{path}' where path refers to the property in the report object.
         *
         * @param exceptionMessageFormat message format
         * @return this instance
         */
        public Builder setExceptionMessageFormat(String exceptionMessageFormat) {
            this.exceptionMessageFormat = exceptionMessageFormat;
            return this;
        }

        /**
         * Sets forma for tasks which were stopped.
         * Property access is done by '{path}' where path refers to the property in the report object.
         *
         * @param stopMessageFormat message format
         * @return this instance
         */
        public Builder setStopMessageFormat(String stopMessageFormat) {
            this.stopMessageFormat = stopMessageFormat;
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public Slf4jLogTaskRunListener build() {
            return new Slf4jLogTaskRunListener(this);
        }
    }

    /**
     * Successful tasks which are pushed to the debug mode.
     */
    private Set<String> debugSuccessfulTasks = new HashSet<String>();

    /**
     * Format for the message with success.
     */
    private String successMessageFormat;

    /**
     * Format for the message with exception.
     */
    private String exceptionMessageFormat;

    /**
     * Format for the message with stop.
     */
    private String stopMessageFormat;

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public Slf4jLogTaskRunListener(Builder builder) {
        debugSuccessfulTasks = DomainUtils.softCopyUnmodifiableSet(builder.debugSuccessfulTasks);
        successMessageFormat = builder.successMessageFormat;
        exceptionMessageFormat = builder.exceptionMessageFormat;
        stopMessageFormat = builder.stopMessageFormat;
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardNotEmptyStringInCollection(debugSuccessfulTasks, "debugSuccessfulTasks cannot have empty element");
        ValidationUtils.guardNotEmpty(successMessageFormat, "successMessageFormat cnnot be empty");
        ValidationUtils.guardNotEmpty(exceptionMessageFormat, "exceptionMessageFormat cnnot be empty");
        ValidationUtils.guardNotEmpty(stopMessageFormat, "stopMessageFormat cnnot be empty");
    }

    @Override
    public void runFinishedBySuccess(TaskRunSuccessReport report) {
        if (debugSuccessfulTasks.contains(report.getTaskClass()) && !log.isDebugEnabled()) {
            return;
        }
        String msg = successMessageFormat;
        msg = msg.replace("{code}", report.getCode());
        msg = msg.replace("{taskClass}", report.getTaskClass());
        msg = msg.replace("{parameters}", report.getParameters().toString());
        msg = msg.replace("{result}", report.getResult().toString());
        msg = msg.replace("{startTimestamp}", Dates.format(report.getStartTimestamp(), "yyyy/MM/dd HH:mm:ss"));
        msg = msg.replace("{endTimestamp}", Dates.format(report.getEndTimestamp(), "yyyy/MM/dd HH:mm:ss"));
        msg = msg.replace("{duration}", String.valueOf(report.getDuration()));
        if (debugSuccessfulTasks.contains(report.getTaskClass())) {
            log.debug(msg);
        }
        else {
            log.info(msg);
        }
    }

    @Override
    public void runFinishedByExceptin(TaskRunExceptionReport report) {
        String msg = exceptionMessageFormat;
        msg = msg.replace("{code}", report.getCode());
        msg = msg.replace("{taskClass}", report.getTaskClass());
        msg = msg.replace("{parameters}", report.getParameters().toString());
        msg = msg.replace("{exceptionDetails}", report.getExceptionDetails().toString());
        msg = msg.replace("{startTimestamp}", Dates.format(report.getStartTimestamp(), "yyyy/MM/dd HH:mm:ss"));
        msg = msg.replace("{endTimestamp}", Dates.format(report.getEndTimestamp(), "yyyy/MM/dd HH:mm:ss"));
        msg = msg.replace("{duration}", String.valueOf(report.getDuration()));
        log.error(msg);
    }

    @Override
    public void runFinishedByStop(TaskRunStopReport report) {
        String msg = stopMessageFormat;
        msg = msg.replace("{code}", report.getCode());
        msg = msg.replace("{taskClass}", report.getTaskClass());
        msg = msg.replace("{parameters}", report.getParameters().toString());
        msg = msg.replace("{startTimestamp}", Dates.format(report.getStartTimestamp(), "yyyy/MM/dd HH:mm:ss"));
        msg = msg.replace("{endTimestamp}", Dates.format(report.getEndTimestamp(), "yyyy/MM/dd HH:mm:ss"));
        msg = msg.replace("{duration}", String.valueOf(report.getDuration()));
        log.warn(msg);
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }
}
