package com.enterprisemath.utils.engine;

import org.apache.commons.lang.builder.ToStringBuilder;

import com.enterprisemath.utils.ValidationUtils;
import com.enterprisemath.utils.messaging.SystemMessenger;

/**
 * Implementation of task run listener which sends messages with exception to administrator.
 *
 * @author radek.hecl
 */
public class MessageExceptionTaskRunListener implements TaskRunListener {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Messenger.
         */
        private SystemMessenger messenger;

        /**
         * Sets messenger.
         *
         * @param messenger messenger
         * @return this instance
         */
        public Builder setMessenger(SystemMessenger messenger) {
            this.messenger = messenger;
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return create object
         */
        public MessageExceptionTaskRunListener build() {
            return new MessageExceptionTaskRunListener(this);
        }
    }

    /**
     * Messenger.
     */
    private SystemMessenger messenger;

    /**
     * Creates new instance.
     *
     * @param builder object
     */
    public MessageExceptionTaskRunListener(Builder builder) {
        messenger = builder.messenger;
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardNotNull(messenger, "messenger cannot be null");
    }

    @Override
    public void runFinishedBySuccess(TaskRunSuccessReport report) {
        // nothing to do here
    }

    @Override
    public void runFinishedByExceptin(TaskRunExceptionReport report) {
        messenger.sendError("Error during task " + report.getCode(),
                "Exception has been thrown during invoking " + report.getTaskClass() + " with " + report.getParameters(),
                report.getExceptionDetails());
    }

    @Override
    public void runFinishedByStop(TaskRunStopReport report) {
        // nothing to do here
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

}
