package com.enterprisemath.utils.engine;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.builder.ToStringBuilder;

import com.enterprisemath.utils.DomainUtils;
import com.enterprisemath.utils.ValidationUtils;

/**
 * Listener for task runs which invokes other listeners in ordered manner.
 *
 * @author radek.hecl
 */
public class ListTaskRunListener implements TaskRunListener {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Listeners which are sequentially invoked.
         */
        private List<TaskRunListener> listeners = new ArrayList<TaskRunListener>();

        /**
         * Sets listeners.
         *
         * @param listeners listeners
         * @return this instance
         */
        public Builder setListeners(List<TaskRunListener> listeners) {
            this.listeners = DomainUtils.softCopyList(listeners);
            return this;
        }

        /**
         * Adds listener to the queue.
         *
         * @param listener listener
         * @return this instance
         */
        public Builder addListener(TaskRunListener listener) {
            this.listeners.add(listener);
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public ListTaskRunListener build() {
            return new ListTaskRunListener(this);
        }

    }

    /**
     * Listeners which are sequentially invoked.
     */
    private List<TaskRunListener> listeners;

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public ListTaskRunListener(Builder builder) {
        listeners = DomainUtils.softCopyUnmodifiableList(builder.listeners);
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardNotNullCollection(listeners, "listeners cannot have null element");
    }

    @Override
    public void runFinishedBySuccess(TaskRunSuccessReport report) {
        for (TaskRunListener listener : listeners) {
            listener.runFinishedBySuccess(report);
        }
    }

    @Override
    public void runFinishedByExceptin(TaskRunExceptionReport report) {
        for (TaskRunListener listener : listeners) {
            listener.runFinishedByExceptin(report);
        }
    }

    @Override
    public void runFinishedByStop(TaskRunStopReport report) {
        for (TaskRunListener listener : listeners) {
            listener.runFinishedByStop(report);
        }
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

    /**
     * Creates new instance.
     * 
     * @param listeners listeners
     * @return created instance
     */
    public static ListTaskRunListener create(List<TaskRunListener> listeners) {
        return new ListTaskRunListener.Builder().
                setListeners(listeners).
                build();
    }
}
