package com.enterprisemath.utils.engine;

import com.enterprisemath.utils.DomainUtils;
import com.enterprisemath.utils.ValidationUtils;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;

/**
 * Definition of the engine status.
 *
 * @author radek.hecl
 */
public class EngineStatus {

    /**
     * Builder object.
     */
    public static class Builder {

        /**
         * Task statuses.
         */
        private List<EngineTaskStatus> taskStatuses = new ArrayList<EngineTaskStatus>();

        /**
         * Sets task statuses.
         *
         * @param taskStatuses task statuses
         * @return this instance
         */
        public Builder setTaskStatuses(List<EngineTaskStatus> taskStatuses) {
            this.taskStatuses = DomainUtils.softCopyList(taskStatuses);
            return this;
        }

        /**
         * Adds task status.
         *
         * @param taskStatus task status
         * @return this instance
         */
        public Builder addTaskStatus(EngineTaskStatus taskStatus) {
            this.taskStatuses.add(taskStatus);
            return this;
        }

        /**
         * Builds the result object.
         *
         * @return created object
         */
        public EngineStatus build() {
            return new EngineStatus(this);
        }
    }

    /**
     * Task statuses.
     */
    private List<EngineTaskStatus> taskStatuses;

    /**
     * Creates new instance.
     *
     * @param builder builder object
     */
    public EngineStatus(Builder builder) {
        taskStatuses = DomainUtils.softCopyUnmodifiableList(builder.taskStatuses);
        guardInvariants();
    }

    /**
     * Guards this object to be consistent. Throws exception if this is not the case.
     */
    private void guardInvariants() {
        ValidationUtils.guardNotNullCollection(taskStatuses, "taskStatuses cannot have null element");
    }

    /**
     * Returns statuses of all running tasks.
     * 
     * @return statuses of all running tasks
     */
    public List<EngineTaskStatus> getTaskStatuses() {
        return taskStatuses;
    }

    @Override
    public int hashCode() {
        return HashCodeBuilder.reflectionHashCode(this);
    }

    @Override
    public boolean equals(Object obj) {
        return EqualsBuilder.reflectionEquals(this, obj);
    }

    @Override
    public String toString() {
        return ToStringBuilder.reflectionToString(this);
    }

}
