package com.enterprisemath.utils;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.lang.StringUtils;

/**
 * Utility class for encoding / decoding objects to / from the string maps.
 * Transformations are supported for the following data types:
 * <ul>
 * <li>Null</li>
 * <li>Primitives (raw and boxed) - Boolean, Byte, Integer, Long, Float, Double</li>
 * <li>Strings</li>
 * <li>Collections - lists, sets and maps without specified implementation</li>
 * <li>Enumerations - stored by name</li>
 * <li>Date - stored in 'yyyy/MM/dd HH:mm:ss.SSS' format without time zone</li>
 * <li>
 * Objects with builder class - builder class must be public static inner class named Builder and has to contain method build.
 * Properties with the same name as in the builder class are serialized. During deserialization they
 * are injected by the setter method (e.g. setProperty(T x)...) or directly if there is no setter method.
 * Properties might be sorted map or sorted set. In such case the sorting is expected to be natural and some standard implementation
 * is used (e.g. TreeSet and TreeMap).
 * </li>
 * </ul>
 *
 * @author radek.hecl
 */
public class StringMapCodec {

    /**
     * Set of primitive classes.
     */
    private static final Set<Class<?>> PRIMITIVE_CLASSES = DomainUtils.<Class<?>>createSet(
            boolean.class, byte.class, int.class, long.class, float.class, double.class,
            Boolean.class, Byte.class, Integer.class, Long.class, Float.class, Double.class, String.class);

    /**
     * Prevents construction.
     */
    private StringMapCodec() {
    }

    /**
     * Encodes object to the string map.
     *
     * @param object input object
     * @return map with encoded object
     */
    public static Map<String, String> encode(Object object) {
        return encode(object, "");
    }

    /**
     * Encodes object to the string map.
     *
     * @param object input object
     * @param prefix prefix to use for the map keys
     * @return map with encoded object
     */
    public static Map<String, String> encode(Object object, String prefix) {
        if (prefix == null) {
            prefix = "";
        }
        else if (StringUtils.isNotEmpty(prefix) && !prefix.endsWith(".")) {
            prefix = prefix + ".";
        }
        if (object == null) {
            return Collections.singletonMap(prefix + "class", "null");
        }
        else if (PRIMITIVE_CLASSES.contains(object.getClass())) {
            Map<String, String> res = new TreeMap(PropertyStringComparator.create());
            res.put(prefix + "class", object.getClass().getName());
            res.put(prefix + "value", object.toString());
            return res;
        }
        else if (object instanceof List) {
            Map<String, String> res = new TreeMap(PropertyStringComparator.create());
            res.put(prefix + "class", List.class.getName());
            List<?> list = (List<?>) object;
            int i = 0;
            for (Object item : list) {
                String prfx = prefix + "items[" + i + "]";
                res.putAll(encode(item, prfx));
                ++i;
            }
            return res;
        }
        else if (object instanceof Set) {
            Map<String, String> res = new TreeMap(PropertyStringComparator.create());
            res.put(prefix + "class", Set.class.getName());
            Set<?> set = (Set<?>) object;
            int i = 0;
            for (Object item : set) {
                String prfx = prefix + "items[" + i + "]";
                res.putAll(encode(item, prfx));
                ++i;
            }
            return res;
        }
        else if (object instanceof Map) {
            Map<String, String> res = new TreeMap(PropertyStringComparator.create());
            res.put(prefix + "class", Map.class.getName());
            Map<?, ?> map = (Map<?, ?>) object;
            int i = 0;
            for (Map.Entry<?, ?> entry : map.entrySet()) {
                String kprfx = prefix + "entries[" + i + "].key";
                res.putAll(encode(entry.getKey(), kprfx));
                String vprfx = prefix + "entries[" + i + "].value";
                res.putAll(encode(entry.getValue(), vprfx));
                ++i;
            }
            return res;
        }
        else if (object instanceof Enum) {
            Enum enm = (Enum) object;
            return DomainUtils.createMap(prefix + "class", object.getClass().getName(), prefix + "value", enm.name());
        }
        else if (object instanceof Date) {
            Date date = (Date) object;
            return DomainUtils.createMap(prefix + "class", "java.util.Date", prefix + "value", Dates.format(date, "yyyy/MM/dd HH:mm:ss.SSS"));
        }
        else {
            Class<?> builderClass = getBuilderClass(object.getClass().getName());
            SortedSet<String> properties = getPropertyNames(builderClass);
            Map<String, String> res = new TreeMap(PropertyStringComparator.create());
            res.put(prefix + "class", object.getClass().getName());
            for (String property : properties) {
                Object propValue = getPropertyValue(object, property);
                String prfx = prefix + property;
                res.putAll(encode(propValue, prfx));
            }
            return res;
        }
    }

    /**
     * Decodes object from the string map.
     *
     * @param <T> object type
     * @param buf data buffer
     * @param clazz desired class after extraction
     * @return decoded object
     */
    public static <T> T decode(Map<String, String> buf, Class<T> clazz) {
        return decode(buf, "", clazz);
    }

    /**
     * Decodes object from the string map.
     *
     * @param <T> object type
     * @param buf data buffer
     * @param prefix prefix to use for the map keys
     * @param clazz desired class after extraction
     * @return decoded object
     */
    public static <T> T decode(Map<String, String> buf, String prefix, Class<T> clazz) {
        if (prefix == null) {
            prefix = "";
        }
        else if (StringUtils.isNotEmpty(prefix) && !prefix.endsWith(".")) {
            prefix = prefix + ".";
        }
        String className = buf.get(prefix + "class");
        if (className.equals("null")) {
            return null;
        }
        Class<?> objectType = getClass(className);
        if (PRIMITIVE_CLASSES.contains(objectType)) {
            String val = buf.get(prefix + "value");
            return (T) getPrimitiveValue(objectType, val);
        }
        else if (objectType.equals(List.class)) {
            List<Object> res = new ArrayList<Object>();
            int i = 0;
            while (buf.containsKey(prefix + "items[" + i + "].class")) {
                String prfx = prefix + "items[" + i + "]";
                Object elm = decode(buf, prfx, Object.class);
                res.add(elm);
                ++i;
            }
            return (T) res;
        }
        else if (objectType.equals(Set.class)) {
            Set<Object> res = new HashSet<Object>();
            int i = 0;
            while (buf.containsKey(prefix + "items[" + i + "].class")) {
                String prfx = prefix + "items[" + i + "]";
                Object elm = decode(buf, prfx, Object.class);
                res.add(elm);
                ++i;
            }
            return (T) res;
        }
        else if (objectType.equals(Map.class)) {
            Map<Object, Object> res = new HashMap<Object, Object>();
            int i = 0;
            while (buf.containsKey(prefix + "entries[" + i + "].key.class")) {
                String kprfx = prefix + "entries[" + i + "].key";
                String vprfx = prefix + "entries[" + i + "].value";
                Object key = decode(buf, kprfx, Object.class);
                Object value = decode(buf, vprfx, Object.class);
                res.put(key, value);
                ++i;
            }
            return (T) res;
        }
        else if (objectType.isEnum()) {
            String val = buf.get(prefix + "value");
            return (T) createEnum(objectType, val);
        }
        else if (objectType.equals(Date.class)) {
            String val = buf.get(prefix + "value");
            return (T) Dates.parse(val, "yyyy/MM/dd HH:mm:ss.SSS");
        }
        else {
            Class<?> builderClass = getBuilderClass(objectType.getName());
            Set<String> properties = getPropertyNames(builderClass);
            Object builder = createObject(builderClass);
            for (String property : properties) {
                String prfx = prefix + property;
                Object val = decode(buf, prfx, Object.class);
                setPropertyValue(builder, property, val);
            }
            return (T) buildObject(builder);
        }
    }

    /**
     * Returns builder class for the object.
     *
     * @param objClassName object class name
     * @return builder class
     */
    private static Class<?> getBuilderClass(String objClassName) {
        try {
            Class<?> builderClass = Class.forName(objClassName + "$Builder");
            return builderClass;
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException("builder class is not defined, this is not supported: " + objClassName, e);
        }
    }

    /**
     * Returns class for the object.
     *
     * @param className class name
     * @return builder class
     */
    private static Class<?> getClass(String className) {
        try {
            Class<?> builderClass = Class.forName(className);
            return builderClass;
        } catch (ClassNotFoundException e) {
            throw new IllegalArgumentException("class not found: " + className, e);
        }
    }

    /**
     * Returns property names.
     *
     * @param clazz class to inspect
     * @return property names
     */
    private static SortedSet<String> getPropertyNames(Class<?> clazz) {
        SortedSet<String> res = new TreeSet<String>();
        Field[] fields = clazz.getDeclaredFields();
        if (fields == null) {
            return res;
        }
        for (Field field : fields) {
            res.add(field.getName());
        }
        return res;
    }

    /**
     * Creates object.
     *
     * @param clazz class
     * @return created object
     */
    private static Object createObject(Class<?> clazz) {
        try {
            Constructor<?> constructor = clazz.getConstructor();
            Object res = constructor.newInstance();
            return res;
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        } catch (SecurityException e) {
            throw new RuntimeException(e);
        } catch (InstantiationException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e);
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Creates enumeration.
     *
     * @param clazz enumeration class
     * @param value string representation
     * @return created enumeration
     */
    private static Object createEnum(Class<?> clazz, String value) {
        try {
            Method method = clazz.getMethod("valueOf", String.class);
            Object res = method.invoke(null, value);
            return res;
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        } catch (SecurityException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e);
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Builds the object.
     *
     * @param builderObject builder object
     * @return created object
     */
    private static Object buildObject(Object builderObject) {
        try {
            Method method = builderObject.getClass().getMethod("build");
            Object res = method.invoke(builderObject);
            return res;
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        } catch (SecurityException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e);
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Returns property class.
     *
     * @param obj object
     * @param property property name
     * @return property class
     */
    private static Class<?> getPropertyClass(Object obj, String property) {
        try {
            Field field = obj.getClass().getDeclaredField(property);
            return field.getType();
        } catch (NoSuchFieldException e) {
            throw new RuntimeException(e);
        } catch (SecurityException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Returns property value.
     *
     * @param obj object
     * @param property property name
     * @return property class
     */
    private static Object getPropertyValue(Object obj, String property) {
        try {
            Field field = obj.getClass().getDeclaredField(property);
            if (field.isAccessible()) {
                return field.get(obj);
            }
            else {
                field.setAccessible(true);
                Object res = field.get(obj);
                field.setAccessible(false);
                return res;
            }
        } catch (NoSuchFieldException e) {
            throw new RuntimeException(e);
        } catch (SecurityException e) {
            throw new RuntimeException(e);
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Sets value of the property.
     *
     * @param obj object to set it in
     * @param property property name
     * @param value property value
     */
    private static void setPropertyValue(Object obj, String property, Object value) {
        if (setPropertyValueBySetter(obj, property, value)) {
            return;
        }
        if (setPropertyValueByInjection(obj, property, value)) {
            return;
        }
        throw new RuntimeException("Unable to set property value: obj = " + obj + "; property = " + property + "; value = " + value);
    }

    /**
     * Sets value of the property by the setter.
     *
     * @param obj object to set it in
     * @param property property name
     * @param value property value
     * @return true if value was set, false otherwise
     */
    private static boolean setPropertyValueBySetter(Object obj, String property, Object value) {
        String setterName = "set" + StringUtils.capitalize(property);
        Class<?> clazz = obj.getClass();
        Method[] methods = clazz.getMethods();
        Method setter = null;
        for (Method candidate : methods) {
            if (!candidate.getName().equals(setterName)) {
                continue;
            }
            Class<?>[] parameters = candidate.getParameterTypes();
            if (parameters.length != 1) {
                continue;
            }
            if (value == null) {
                setter = candidate;
            }
            else {
                if (parameters[0].isPrimitive()) {
                    if (parameters[0].equals(double.class) && value.getClass().equals(Double.class) ||
                            parameters[0].equals(float.class) && value.getClass().equals(Float.class) ||
                            parameters[0].equals(long.class) && value.getClass().equals(Long.class) ||
                            parameters[0].equals(int.class) && value.getClass().equals(Integer.class) ||
                            parameters[0].equals(byte.class) && value.getClass().equals(Byte.class) ||
                            parameters[0].equals(boolean.class) && value.getClass().equals(Boolean.class)) {
                        setter = candidate;
                        continue;
                    }
                }
                if (parameters[0].isAssignableFrom(SortedSet.class) && value.getClass().equals(HashSet.class)) {
                    value = new TreeSet((Set) value);
                    setter = candidate;
                }
                if (parameters[0].isAssignableFrom(SortedMap.class) && value.getClass().equals(HashMap.class)) {
                    value = new TreeMap((Map) value);
                    setter = candidate;
                }
                if (!parameters[0].isAssignableFrom(value.getClass())) {
                    continue;
                }
                setter = candidate;
            }
        }
        if (setter == null) {
            return false;
        }
        try {
            setter.invoke(obj, value);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e);
        } catch (InvocationTargetException e) {
            throw new RuntimeException(e);
        }
        return true;
    }

    /**
     * Sets value of the property by injection directly to the object.
     *
     * @param obj object to set it in
     * @param property property name
     * @param value property value
     * @return true if value was set, false otherwise
     */
    private static boolean setPropertyValueByInjection(Object obj, String property, Object value) {
        Class<?> clazz = obj.getClass();
        Field field = null;
        try {
            field = clazz.getDeclaredField(property);
        } catch (NoSuchFieldException e) {
            return false;
        } catch (SecurityException e) {
            throw new RuntimeException(e);
        }

        try {
            Class<?> fc = field.getType();
            Class<?> vc = value.getClass();
            if (fc.isAssignableFrom(SortedSet.class) && !fc.isAssignableFrom(Set.class) && vc.equals(HashSet.class)) {
                value = new TreeSet((Set) value);
            }
            if (fc.isAssignableFrom(SortedMap.class) && !fc.isAssignableFrom(Map.class) && vc.equals(HashMap.class)) {
                value = new TreeMap((Map) value);
            }

            if (field.isAccessible()) {
                field.set(obj, value);
            }
            else {
                field.setAccessible(true);
                field.set(obj, value);
                field.setAccessible(false);
            }
        } catch (IllegalArgumentException e) {
            throw new RuntimeException(e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException(e);
        }

        return true;
    }

    /**
     * Returns value of primitive object.
     *
     * @param clazz primitive class
     * @param val string value representation
     * @return value
     */
    private static Object getPrimitiveValue(Class<?> clazz, String val) {
        if (clazz.equals(String.class)) {
            return val;
        }
        else if (clazz.equals(Double.class) || clazz.equals(double.class)) {
            return Double.valueOf(val);
        }
        else if (clazz.equals(Float.class) || clazz.equals(float.class)) {
            return Float.valueOf(val);
        }
        else if (clazz.equals(Long.class) || clazz.equals(long.class)) {
            return Long.valueOf(val);
        }
        else if (clazz.equals(Integer.class) || clazz.equals(int.class)) {
            return Integer.valueOf(val);
        }
        else if (clazz.equals(Byte.class) || clazz.equals(byte.class)) {
            return Byte.valueOf(val);
        }
        else if (clazz.equals(Boolean.class) || clazz.equals(boolean.class)) {
            return Boolean.valueOf(val);
        }
        else {
            throw new RuntimeException("unknown primitive type class: " + clazz);
        }
    }

}
