package com.enterprisemath.utils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;

/**
 * Utilities to work with JSON.
 *
 * @author radek.hecl
 */
public class JSONUtils {

    /**
     * Prevents construction.
     */
    private JSONUtils() {
    }

    /**
     * Encodes string map as a JSON object.
     * Map is encoded as an array of entries, naturally ordered by key as a property.
     * For example map {"hello" : "world", "hey" : "baby"} is encoded as [{"key" : "hello", "value" : "world"}, {"key" : "hey", "value" : "baby"}].
     * This allows any form of key to be captured.
     *
     * @param data data
     * @return encoded string
     */
    public static String encodeStringMap(Map<String, String> data) {
        SortedMap<String, String> sorted = new TreeMap<String, String>(PropertyStringComparator.create());
        sorted.putAll(data);
        List<JSONObject> items = new ArrayList<JSONObject>();
        for (String key : sorted.keySet()) {
            JSONObject item = new JSONObject();
            item.put("key", key);
            item.put("value", sorted.get(key));
            items.add(item);
        }
        JSONArray array = new JSONArray(items);
        return array.toString();
    }

    /**
     * Encodes string map as a JSON object in the simple way.
     * Map is encoded as it is. For example {"hello" : "world", "hey" : "baby"}.
     * This method provides more compact result with precondition that the format of the keys doesn't break any convention.
     * For example in some systems it might cause troubles to have a key of the property style format like "items[0].size".
     *
     * @param data data
     * @return encoded string
     */
    public static String encodeSimpleStringMap(Map<String, String> data) {
        SortedMap<String, String> sorted = DomainUtils.softCopySortedMap(data);
        JSONObject map = new JSONObject(sorted);
        return map.toString();
    }

    /**
     * Decodes string map from the JSON object. Accepts both type of strings - map encoded as an array and as a simple format.
     *
     * @param data data
     * @return decoded map
     */
    public static Map<String, String> decodeStringMap(String data) {
        Map<String, String> res = new HashMap<String, String>();
        if (data.startsWith("[")) {
            JSONArray array = new JSONArray(data);
            for (int i = 0; i < array.length(); ++i) {
                JSONObject item = (JSONObject) array.get(i);
                String key = item.getString("key");
                String value = item.getString("value");
                res.put(key, value);
            }
        }
        else if (data.startsWith("{")) {
            JSONObject obj = new JSONObject(data);
            for (String key : obj.keySet()) {
                res.put(key, obj.getString(key));
            }
        }
        else {
            throw new IllegalArgumentException("data string must start with '[' or '{'" + StringUtils.substring(data, 0, 4) + "...");
        }
        return res;
    }

    /**
     * Encodes string list as a JSON object
     *
     * @param data data
     * @return encoded string
     */
    public static String encodeStringList(List<String> data) {
        JSONArray res = new JSONArray();
        for (String elm : data) {
            res.put(elm);
        }
        return res.toString();
    }

    /**
     * Decodes string list from the JSON object.
     *
     * @param data data
     * @return decoded map
     */
    public static List<String> decodeStringList(String data) {
        List<String> res = new ArrayList<String>();
        JSONArray array = new JSONArray(data);
        for (int i = 0; i < array.length(); ++i) {
            String item = (String) array.get(i);
            res.add(item);
        }
        return res;
    }

}
